package directory

import (
	"path/filepath"
	"strings"
)

import (
	"io/ioutil"
	"log"
)

// GoRepos returns all repos that are related to go. See IsGoRepo for
func GoRepos(root string) ([]string, error) {
	repoDirectories, err := Repos(root)
	if err != nil {
		log.Printf("Could not retrieve repos %v", err.Error())
		return nil, err
	}
	var goRepos []string
	for _, repo := range repoDirectories {
		isGoRepo, err := IsGoRepo(repo)
		if err != nil {
			log.Printf("Could not filter go repo %v", repo)
			return nil, err
		}
		if isGoRepo {
			goRepos = append(goRepos, repo)
		}
	}
	return goRepos, nil
}

// ListRepos goes through the directory of github repos downloaded by fetchall from common/meta and returns the home
// directory of each file.
func Repos(root string) ([]string, error) {
	var repoDirectories []string
	orgs, err := ioutil.ReadDir(root)
	if err != nil {
		log.Printf("Could not read directory %v", root)
		return nil, err
	}
	for _, org := range orgs {
		orgName := org.Name()
		if !strings.Contains(orgName, ".") && org.IsDir() {
			repo := filepath.Join(root, orgName)
			repos, err := ioutil.ReadDir(repo)
			if err != nil {
				log.Printf("Could not read directory %v", repo)
				return nil, err
			}
			for _, r := range repos {
				if !strings.Contains(r.Name(), ".") {
					repoPath := filepath.Join(root, orgName, r.Name())
					repoDirectories = append(repoDirectories, repoPath)
				}
			}
		}
	}
	return repoDirectories, nil
}

// GO_DEPENDENCY_FILES are top level repo files that indicate if a repo is go related
// Note that Godeps is a folder and if a go source root is nested, this will not find it
// This is mostly true for repos that have multiple languages supported.
var GO_DEPENDENCY_FILES = map[string]bool{
	"glide.yaml": true, "Godeps": true, "Gopkg.toml": true,
}

// IsGoRepo takes a repo directory locations and returns true if it is a go Repo
// Currently, this is done by looking for either glide, godep, or dep dependency files.
func IsGoRepo(repo string) (bool, error) {
	topLevelFiles, err := ioutil.ReadDir(repo)
	if err != nil {
		return false, err
	}

	for _, file := range topLevelFiles {
		if GO_DEPENDENCY_FILES[file.Name()] {
			return true, nil
		}
	}

	return false, nil
}
