package resolvers

import (
	"encoding/json"
	"errors"
	"fmt"
	"github.com/Masterminds/glide/cfg"
	"github.com/pelletier/go-toml"
	"io/ioutil"
	"log"
	"path/filepath"
)

var GO_DEPENDENCY_FILES = map[string]func(string) ([]string, error){
	"glide.yaml": ParseGlide, "Godeps": ParseGodep, "Gopkg.toml": ParseDep,
}

var RESOLUTION_ORDER = []string{"Gopkg.toml", "glide.yaml", "Godeps"}

// ParseDependencies looks at a golang repo and tries to extract a list of direct dependencies
// It tries to avoid any transitive dependencies
func ParseDependencies(repoDirectory string) ([]string, error) {
	topLevelFiles, err := ioutil.ReadDir(repoDirectory)
	if err != nil {
		log.Printf("Failed to open repo %v because of %v", repoDirectory, err.Error())
		return nil, err
	}

	// There are some repos that will have leftover GoDep folders that are actually using glide.yaml
	var matchingDependencies []string
	for _, file := range topLevelFiles {
		fileName := file.Name()
		if depResolver := GO_DEPENDENCY_FILES[fileName]; depResolver != nil {
			matchingDependencies = append(matchingDependencies, fileName)
		}
	}
	if len(matchingDependencies) == 0 {
		return nil, errors.New(
			fmt.Sprintf("Could not find any go dependency files inside of repo %v", repoDirectory))
	}
	for _, depOrder := range RESOLUTION_ORDER {
		for _, depFile := range matchingDependencies {
			if depFile == depOrder {
				depPath := filepath.Join(repoDirectory, depFile)
				return GO_DEPENDENCY_FILES[depFile](depPath)
			}
		}
	}
	// For some reason, we could not match found dependency go files to our list of order
	return nil, errors.New("Something went wrong, this should never be reached")
}

// ParseGodep looks at a Godeps folder and parses out a list of direct dependencies for the repo
func ParseGodep(godepPath string) ([]string, error) {
	var godepsFilePath = filepath.Join(godepPath, "Godeps.json")
	godepsFile, err := ioutil.ReadFile(godepsFilePath)
	if err != nil {
		// TODO(michang) handle old version of godeps. see https://github.com/tools/godep/blob/master/godepfile.go#L60:6
		log.Printf("Could not open the godeps file %v, is Godeps.json missing?", godepsFile)
		return nil, err
	}

	type Dependency struct {
		ImportPath string
	}

	type DepFile struct {
		Deps []Dependency
	}

	var depFile DepFile
	json.Unmarshal(godepsFile, &depFile)

	var dependencies []string
	for _, d := range depFile.Deps {
		dependencies = append(dependencies, d.ImportPath)
	}

	return dependencies, nil
}

// parseGlide looks at a glide.yaml file and parses out a list of direct dependencies
func ParseGlide(glidePath string) ([]string, error) {
	glideFile, err := ioutil.ReadFile(glidePath)
	if err != nil {
		log.Printf("Could not open the glide file %v, is glide.yaml missing?", glidePath)
		return nil, err
	}

	glideConfig, err := cfg.ConfigFromYaml(glideFile)
	if err != nil {
		log.Printf("Unable to parse glide file %v", glidePath)
		return nil, err
	}

	var dependencies []string
	for _, d := range glideConfig.Imports {
		dependencies = append(dependencies, d.Name)
	}

	return dependencies, nil
}

// parseDep looks at a Dep Gopkg.toml file and parses out a list of direct dependencies for the repo
func ParseDep(depPath string) ([]string, error) {
	// TODO(michang) investigate why dep files seem to have very sparse .toml files
	depByte, err := ioutil.ReadFile(depPath)
	if err != nil {
		log.Printf("Could not open the dep file %v, is dep File missing?", depPath)
		return nil, err
	}

	type Dependency struct {
		Name string
	}

	type DepFile struct {
		Constraints []Dependency `toml:"constraint,omitempty"`
	}

	var depFile DepFile
	// Dep has a func called readManifest, but it is not public
	// https://github.com/golang/dep/blob/master/manifest.go#L328
	// Which is why we read from toml directly
	toml.Unmarshal(depByte, &depFile)
	if err != nil {
		log.Printf("Unable to parse dep file %v", depPath)
		return nil, err
	}

	var dependencies []string
	for _, d := range depFile.Constraints {
		dependencies = append(dependencies, d.Name)
	}

	return dependencies, nil
}
