package twitchldap

import (
	"testing"

	"gopkg.in/ldap.v2"
)

type CNTest struct {
	DN         string
	ExpectedCN string
}

// Mock connection implements LDAPSearchable interface
type MockConnection struct {
	uid    string
	cn     string
	active string
	gn     string
	sn     string
}

// Mock method
func (c *MockConnection) Search(sr *ldap.SearchRequest) (*ldap.SearchResult, error) {
	dummyEntry := &ldap.Entry{
		DN: "test",
		Attributes: []*ldap.EntryAttribute{
			{
				Name:   "cn",
				Values: []string{c.cn},
			},
			{
				Name:   "uid",
				Values: []string{c.uid},
			},
			{
				Name:   "employeeNumber",
				Values: []string{"1234567"},
			},
			{
				Name:   "ds-sync-hist",
				Values: []string{"modifyTimestamp:0000016345c55c721a4b0000165c:repl:20180509163912Z"},
			},
			{
				Name:   "ds-sync-hist",
				Values: []string{"ds-pwp-account-disabled:0000016345c55c721a4b0000165c:add:" + c.active},
			},
			{
				Name:   "givenName",
				Values: []string{c.gn},
			},
			{
				Name:   "sn",
				Values: []string{c.sn},
			},
		},
	}
	result := &ldap.SearchResult{
		Entries: []*ldap.Entry{dummyEntry},
	}
	return result, nil
}

// Mock method

func (c *MockConnection) Close() {
}

// Initializer for Mock client
func NewMockClient(uid string, cn string, active string, gn string, sn string) (*LDAPClient, error) {
	return &LDAPClient{
		conn: &MockConnection{
			uid:    uid,
			cn:     cn,
			active: active,
			gn:     gn,
			sn:     sn,
		},
	}, nil
}

func TestExtractCNFromDN(t *testing.T) {
	testCases := []CNTest{
		{
			DN:         `cn=Jonathan Simpson-Bint,ou=Users,dc=justin,dc=tv`,
			ExpectedCN: `Jonathan Simpson-Bint`,
		},
		{
			DN:         `cn=Mark Weiler (mweiler),ou=Users,dc=justin,dc=tv`,
			ExpectedCN: `Mark Weiler (mweiler)`,
		},
		{
			DN:         `cn=BrandonD. Ove,ou=Users,dc=justin,dc=tv`,
			ExpectedCN: `BrandonD. Ove`,
		},
		{
			DN:         `cn=Chris Milward,ou=Users,dc=justin,dc=tv`,
			ExpectedCN: `Chris Milward`,
		},
		{
			DN:         `cn=Jean-Nicolas Vollmer,ou=Users,dc=justin,dc=tv`,
			ExpectedCN: `Jean-Nicolas Vollmer`,
		},
	}

	for _, testCase := range testCases {
		if cn, err := extractCNFromDN(testCase.DN); err != nil || cn != testCase.ExpectedCN {
			t.Fatal("Was unable to parse Username from LDAP. TestCase was", testCase, err)
		}
	}
}

func TestLDAPClient_GetUserInfoByName(t *testing.T) {
	client, err := NewMockClient("emmett", "Emmett Shear", "false", "", "")
	if err != nil {
		t.Fatalf("was unable to initialize client because %s", err.Error())
	}
	defer client.Close()
	user, err := client.GetUserInfoByName("emmett")
	if err != nil {
		t.Fatalf("was unable to get info for emmett because %s", err.Error())
	}
	if user.CN != "Emmett Shear" {
		t.Fatalf("did not get the right cn for emmett")
	}
	if user.Inactive {
		t.Fatalf("did not get the right value for active for emmett")
	}
	client, _ = NewMockClient("imarko", "Istavan Marko", "true", "", "")
	user, err = client.GetUserInfoByName("imarko")
	if err != nil {
		t.Fatalf("was unable to get info for imarko because %s", err.Error())
	}
	if !user.Inactive {
		t.Fatalf("did not get the right value for active for imarko")
	}
	client, _ = NewMockClient("fzaidi", "Syed Zaidi", "true", "Farhan", "Zaidi")
	user, err = client.GetUserInfoByName("fzaidi")
	if err != nil {
		t.Fatalf("was unable to get info for fzaidi because %s", err.Error())
	}
	if user.CN != "Syed Zaidi" {
		t.Fatalf("got wrong CN for fzaidi, got %s instead", user.CN)
	}
	if user.GivenName != "Farhan" {
		t.Fatalf("got wrong givenname for fzaidi, got %s instead", user.GivenName)
	}
	if user.PreferredName != "Farhan Zaidi" {
		t.Fatalf("got wrong preferred name for fzaidi, got %s instead", user.PreferredName)
	}
}

func TestLDAPClient_GetUserInfoInactive(t *testing.T) {
	client, err := NewMockClient("warnold", "Woody Arnold", "true", "", "")
	if err != nil {
		t.Fatalf("was unable to initialize client because %s", err.Error())
	}
	defer client.Close()
	user, err := client.GetUserInfoByName("warnold")
	if err != nil {
		t.Fatalf("was unable to get info for warnold because %s", err.Error())
	}
	if !user.Inactive {
		t.Fatalf("inactive=false for warnold")
	}
	if user.InactiveTime.IsZero() {
		t.Fatalf("inactivatedTime is zero!")
	}
}

func TestLDAPClient_GetUserInfoInactiveNegativeCase(t *testing.T) {
	client, err := NewMockClient("lukemng", "Luke Maung", "false", "", "")
	if err != nil {
		t.Fatalf("was unable to initialize client because %s", err.Error())
	}
	defer client.Close()
	user, err := client.GetUserInfoByName("lukemng")
	if err != nil {
		t.Fatalf("was unable to get info for lukemng because %s", err.Error())
	}
	if user.Inactive {
		t.Fatalf("inactive=true for lukemng")
	}
	if !user.InactiveTime.IsZero() {
		t.Fatalf("inactivatedTime is NOT zero!")
	}
}

// nolint
func DisabledTestRealLDAPClient(t *testing.T) {
	client, err := NewClient()
	if err != nil {
		t.Fatalf("%s", err.Error())
	}
	info, err := client.GetUserInfo(373564) // userid: wademat
	if err != nil {
		t.Fatalf("%s", err.Error())
	}
	t.Logf("inactive=%v", info.Inactive)
	t.Logf("inactive time=%s", info.InactiveTime)
}
