const fs = require('fs');
const mkdirp = require('mkdirp');
const path = require('path');

const browserMobProxy = require('../base/BrowserMobClient');
const testRailFile = require('../utils/TestRailHelper');
const testRailHelper = new testRailFile();
const TEST_RAIL_RUN_ID_FILE = 'tr-run-id.txt';

exports.testRailParams = { 'browser': '', 'suiteId': 0, 'planId': 0, 'type': '' };
exports.testCaseId = 0;
exports.testRunId = 0;
exports.consolidatedResult = [];

exports.config = {
    /**
    * Server Configuration
    */
    host: '0.0.0.0',
    port: 4444,
    maxInstances: 1,
    blackListUrls: [],
    capabilities: [
        {
            browserName: 'firefox',
            'acceptInsecureCerts':true,
            'moz:firefoxOptions': {}
        },
        {
            browserName: 'chrome',
            chromeOptions: {}
        }
    ],

    /**
    * specify test file
    */
    specs: [
        './tests/specs/**/*.js'
    ],

    exclude: [],

    /**
    * test configurations
    */
    logLevel: 'silent',
    coloredLogs: true,
    screenshotPath: 'screenshots',
    baseUrl: '',
    waitforTimeout: 10000,
    connectionRetryCount: 3,
    framework: 'mocha',

    reporters: ['spec', 'junit'],
    reporterOptions: {
        outputDir: './'
    },
    mochaOpts: {
        ui: 'bdd',
        timeout: 70000
    },

    /**
    * hooks
    */
    onPrepare: function (config, capabilities) {
        console.log('let\'s go');
        if (exports.testRailParams.type !== 'local') {
            console.log('===========================================================================================================================');
            console.log('Creating the Test Rail Run');
            let testRun = testRailHelper.addPlanEntry(exports.testRailParams.planId, {
                'suite_id':exports.testRailParams.suiteId,
                'include_all':true,
                "name": `${exports.testRailParams.browser} ${exports.testRailParams.type} run => ${new Date().toJSON().slice(0,16)}`,
                'runs': [
                    {
                        "include_all": true
                    }
                ]
            });
            exports.testRailParams['testRunId'] = JSON.parse(testRun.body).runs[0].id;
            fs.writeFile(TEST_RAIL_RUN_ID_FILE, exports.testRailParams.testRunId, function (err) {
                if (err) throw err;
            });
            console.log(`Test Rail URL ==>> https://testrail.internal.justin.tv/index.php?/runs/view/${exports.testRailParams.testRunId}&group_by=cases:section_id&group_order=asc`);
            console.log('===========================================================================================================================');
        }
    },

    beforeSession: function (config, capabilities, specs) {
        console.log(capabilities)
    },

    before: function (capabilities, specs) {
        // Pause before loading capabilities to avoid cdp mac-4 agent execution timeout
        browser.pause(2000);
        let chai = require('chai');
        expect = chai.expect;
        assert = chai.assert;
        browser.setViewportSize({ height: 1000, width: 1200 }, false);
        browserHelper = require('../base/BrowserHelper');
    },

    /**
    * Function to be executed before a test (in Mocha/Jasmine) or a step (in Cucumber) starts.
    * @param {Object} test test details
    beforeTest: function (test) {
    },
    */

    afterTest: function (test) {
        if (exports.testRailParams.type !== 'local') {
            if (test.passed) {
                exports.consolidatedResult.push({"case_id": exports.testCaseId, 'status_id': 1});
                return;
            }
            //When the test case has failed, update case in the test rail run as failed
            exports.consolidatedResult.push({"case_id": exports.testCaseId, 'status_id': 5});
        }
        // get current test title and clean it, to use it as file name
        let filename = encodeURIComponent(test.title.replace(/[\s+&\/\\#,+()$~%.'"=:*?<>{}]/g, '-'));
        let screenshotFolder = path.join(this.screenshotPath);
        // build file path
        let filePath = path.join(screenshotFolder, '/' + filename + '.png');
        if (!fs.existsSync(screenshotFolder)) {
            mkdirp.mkdirp(screenshotFolder);
        }
        // save screenshot
        browser.saveScreenshot(filePath);
    },

    afterSession: function (config, capabilities, specs) {
        if (exports.testRailParams.type !== 'local') {
            let testRailRunId = parseInt(fs.readFileSync(TEST_RAIL_RUN_ID_FILE), 10);
            testRailHelper.addResultsForCases(testRailRunId, {'results': exports.consolidatedResult});
        }
    },

    onComplete: function (exitCode, config, capabilities) {
        if (exports.testRailParams.type !== 'local') {
            console.log('===========================================================================================================================');
            console.log(`Test Rail URL ==>> https://testrail.internal.justin.tv/index.php?/runs/view/${exports.testRailParams.testRunId}&group_by=cases:section_id&group_order=asc`);
            console.log('===========================================================================================================================');
            fs.unlinkSync(TEST_RAIL_RUN_ID_FILE);
        }
        console.log('All done');
    }
};
