'use strict';

var AWS = require('aws-sdk');
var async = require('async');
var debug = false;

function shouldBroadcast(record) {
  var eventName = record.eventName;
  // @TODO how should we handle delete events.
  if (eventName !== 'INSERT' && eventName !== 'MODIFY') {
    return false;
  }

  var dynamoDBRecord = record.dynamodb;
  var doNotBroadcast = dynamoDBRecord.NewImage.do_not_broadcast.BOOL;
  if (doNotBroadcast === true) {
    return false;
  }

  if (eventName === 'INSERT') {
    return true;
  }

  // safe to imagine the new do_not_broadcast.BOOL is false;
  // But we do not know what has chaged since last time we broadcasted
  // the event. So we would always like to broadcast such event.
  // e.g.
  // 1 User sets do_not_broadcast to true, ( event 1)
  // 2 change the secret, but note do_not_broadcast is still set to true.
  // 3 changes other stuff, note o_not_broadcast is still set to true.
  // ...
  // N user sets do_not_broadcast to false.
  // After these events we cant just rely on last change log, and hence we
  // should broadcast the event.
  var oldDoNotBroadcast = dynamoDBRecord.OldImage.do_not_broadcast.BOOL;
  if (oldDoNotBroadcast === true && doNotBroadcast === false) {
    return true;
  }

  // We would like to broadcast event iff the secret has changed. so that
  // change in category, or any other non critical update do not
  // result in SNS update.
  if (eventName === 'MODIFY') {
    if (dynamoDBRecord.OldImage.value.S !== dynamoDBRecord.NewImage.value.S) {
      return true;
    }
    if (dynamoDBRecord.OldImage.updated_at.N !== dynamoDBRecord.NewImage.updated_at
      .N) {
      return true;
    }
  }
  return false;
}

function buildParams(record) {
  var message = {
    "updated_at": {
      "S": record.dynamodb.NewImage.updated_at.N
    },
    "name": {
      "S": record.dynamodb.NewImage.name.S
    }
  };
  var split = record.eventSourceARN.split('/');
  return {
    Message: JSON.stringify(message),
    TopicArn: 'arn:aws:sns:us-west-2:734326455073:' + split[1]
  };
}

function processEvent(record, cb) {
  console.log('Event: ' + record.eventName + ' Record: ' + record.dynamodb.Keys
    .name.S);
  if (shouldBroadcast(record)) {
    var sns = new AWS.SNS();
    var params = buildParams(record);
    if (debug) {
      console.log('Publishing to SNS: ' + JSON.stringify(params));
    } else {
      console.log('Publishing update for ' + record.dynamodb.NewImage.name.S +
        ' to topic ' + params.TopicArn);
    }
    sns.publish(params, function(err, data) {
      if (err) {
        console.log(err, err.stack);
        return cb(err);
      } else {
        if (debug) {
          console.log("SNS INFO")
          console.log(data);
        }
        return cb();
      }
    });
  } else {
    return cb();
  }
}


exports.handler = function(event, context) {
  if (debug) {
    console.log('Received event:', JSON.stringify(event, null, 2));
  }
  async.each(event.Records, processEvent,
    function(err) {
      if (err) {
        console.log(err, err.stack);
        context.fail({
          processedEvent: false
        });
      } else {
        console.log("Successfully processed " + event.Records.length +
          " records.");
        context.succeed({
          processedEvent: true
        });
      }
    });
};
