package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"os"
	"net"

	"github.com/hashicorp/consul/api"

	"code.justin.tv/common/gonagios"
)

type Config struct {
	Server bool
}

func isServer(fileName string) (bool, error) {
	f, err := os.Open(fileName)
	if err != nil {
		return false, err
	}
	defer f.Close()

	config := Config{}
	if err := json.NewDecoder(f).Decode(&config); err != nil {
		return false, err
	}

	return config.Server, nil
}

func main() {
	nagios := &gonagios.Nagios{}
	var consulConfig string
	var consulHost string

	flag.StringVar(&consulConfig, "config", "/etc/consul.d/main.json", "Consul config file to use")
	flag.StringVar(&consulHost, "consul", "localhost:8500", "Consul addr to check")
	flag.Parse()
	
	config := api.DefaultConfig()
	client, err := api.NewClient(config)

	// Check the current leader:
	leader, err := client.Status().Leader()
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error determaining leader: %v", err),
		)
	}
	if leader == "" {
		nagios.ExitWithStatus(
			gonagios.Critical,
			"No leader elected.",
		)
	}

	// Check the current peers:
	peers, err := client.Status().Peers()
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error determaining peers: %v", err),
		)
	}
	validPeers, err := checkPeers(client, peers)
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error checking peers: %v", err),
		)
	}

	status := gonagios.OK
	if validPeers != len(peers) {
		status = gonagios.Critical
	}

	nagios.ExitWithStatus(status, fmt.Sprintf("%d of %d peers online", validPeers, len(peers)))

}

// checkPeers will take a peer list and return the number of hosts that are
// online.
func checkPeers(client *api.Client, peers []string) (int, error) {
	healthyPeers := 0
	peerSet := map[string]bool{}

	for _, peer := range peers {
		host, _, err := net.SplitHostPort(peer)
		if err != nil {
			return 0, err
		}
		peerSet[host] = true
	}

	nodes, _, err := client.Catalog().Nodes(nil)
	if err != nil {
		return 0, err
	}

	for _, node := range nodes {
		if _, ok := peerSet[node.Address]; ok {
			passing, err := nodePassing(client, node.Node)
			if err != nil {
				return 0, nil
			}
			if passing {
				healthyPeers += 1
			}
		}
	}

	return healthyPeers, nil
}
func nodePassing(client *api.Client, nodeName string) (bool, error) {
	checks, _, err := client.Health().Node(nodeName, nil)
	if err != nil {
		return false, err
	}
	for _, check := range checks {
		if check.CheckID == "serfHealth" && check.Status == "passing" {
			return true, nil
		}
	}
	return false, nil
}