package main

import (
	"flag"
	"fmt"
	"net/http"
	"net/url"
	"path"

	"code.justin.tv/common/gonagios"
)

func main() {
	var node, service, env, consulServer string

	flag.StringVar(&node, "node", node, "node to query")
	flag.StringVar(&service, "service", service, "service to query")
	flag.StringVar(&env, "env", env, "env to query")
	flag.StringVar(
		&consulServer,
		"api",
		"http://localhost:8500/",
		"consul api server to talk to",
	)
	flag.Parse()

	nagios := gonagios.NewNagios()

	if node == "" {
		nagios.ExitWithStatus(gonagios.Unknown, "No node passed to check")
	}

	// If either service or env is set then both service and env must be set.
	if (service != "" || env != "") && !(service != "" && env != "") {
		nagios.ExitWithStatus(
			gonagios.Unknown,
			"Both env and service must be set to check if a service is enabled",
		)
	}

	exists, err := checkKVExistance(
		consulServer,
		path.Join("/v1/kv/node/", node, "down"),
	)
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Unknown,
			fmt.Sprintf("Unable to retreive down file: %v", err),
		)
	}

	if exists {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Node %s is disabled", node),
		)
	}

	if service != "" {
		exists, err := checkKVExistance(
			consulServer,
			path.Join("/v1/kv/env/", env, service, "node", node, "down"),
		)
		if err != nil {
			nagios.ExitWithStatus(
				gonagios.Unknown,
				fmt.Sprintf("Unable to retreive down file for node and service: %v", err),
			)
		}

		if exists {
			nagios.ExitWithStatus(
				gonagios.Critical,
				fmt.Sprintf("Service %s/%s is disabled on node %s", env, service, node),
			)
		}
	}

	nagios.Exit("Not disabled")
}

func checkKVExistance(server, kv string) (bool, error) {
	uri, err := url.Parse(server)
	if err != nil {
		return false, err
	}

	uri.Path = kv
	uri.RawQuery = "raw"

	res, err := http.Get(uri.String())
	if err != nil {
		return false, nil
	}
	defer res.Body.Close()

	if res.StatusCode == 200 {
		return true, nil
	} else if res.StatusCode == 404 {
		return false, nil
	}

	return false, fmt.Errorf("Unknown status code: %d", res.StatusCode)
}
