// consul-alerts will pull information from consul and convert it into nagios alerts
package main

import (
	"flag"
	"fmt"
	"strings"

	"code.justin.tv/common/gonagios"
	"github.com/hashicorp/consul/api"
)

var (
	fqdn          string
	consulServer  string = "localhost:8500"
	defaultSuffix string = "justin.tv"
	suffixes             = []string{"justin.tv", "twitch.tv"}
)

func main() {
	flag.StringVar(&fqdn, "fqdn", fqdn, "fqdn to check")
	flag.StringVar(&consulServer, "api", consulServer, "consul api server to talk to")
	flag.Parse()

	nagios := gonagios.NewNagios()

	if fqdn == "" {
		nagios.ExitWithStatus(gonagios.Critical, fmt.Errorf("Must define --fqdn to check."))
	}

	config := api.DefaultConfig()
	config.Address = consulServer
	client, err := api.NewClient(config)
	if err != nil {
		nagios.ExitWithStatus(gonagios.Critical, err)
	}

	fqdn = AddSuffix(fqdn)
	query := &api.QueryOptions{Datacenter: GetDatacenter(fqdn), AllowStale: true}

	nodeinfoServices, _, err := client.Catalog().Service("nodeinfo", "fqdn="+fqdn, query)
	if err != nil {
		nagios.ExitWithStatus(gonagios.Critical, err)
	}

	if len(nodeinfoServices) > 1 {
		nodeNames := []string{}

		for _, i := range nodeinfoServices {
			nodeNames = append(nodeNames, i.Node)
		}

		nagios.ExitWithStatus(gonagios.Critical,
			fmt.Errorf("consul reports multiple nodes: %v", strings.Join(nodeNames, ", ")))
	}

	if len(nodeinfoServices) == 0 {
		nagios.ExitWithStatus(gonagios.Critical, "consul can't find node")
	}

	nodeHealth, _, err := client.Health().Node(nodeinfoServices[0].Node, query)
	if err != nil {
		nagios.ExitWithStatus(gonagios.Critical, err)
	}

	for _, check := range nodeHealth {
		if check.CheckID == "serfHealth" && check.Status == "passing" {
			nagios.ExitWithStatus(gonagios.OK, "consul reports node is alive")
		}
	}

	nagios.ExitWithStatus(gonagios.Critical, "consul reports node offline")
}

// AddSuffix is a small helper function to check if a fqdn has the .justin.tv
// suffix and add it if not.
func AddSuffix(fqdn string) string {
	hasSuffix := false
	for _, suffix := range suffixes {
		if ok := strings.HasSuffix(fqdn, suffix); ok {
			hasSuffix = true
			break
		}
	}

	if !hasSuffix {
		return fmt.Sprintf("%s.%s", fqdn, defaultSuffix)
	}

	return fqdn
}

// GetDatacenter will get the datacenter.
func GetDatacenter(fqdn string) string {
	split := strings.Split(AddSuffix(fqdn), ".")

	if len(split) <= 3 {
		return "sfo01"
	}

	return split[len(split)-3]
}
