/*
 * consul-alert-server is a nagios check for monitoring the health of consul
 * servers.
 *
 * Goals:
 * - Is the current host a consul server or client.
 *   + If a client return OK.
 * - Are all of the servers in a cluster online
 * - Is a leader selected
 * - Is the current server a member of the WAN
 * - Can it successfully talk to another DC.
 *
 */
package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"os"

	"github.com/hashicorp/consul/api"

	"code.justin.tv/common/gonagios"
)

type Config struct {
	Server bool
}

func isServer(fileName string) (bool, error) {
	f, err := os.Open(fileName)
	if err != nil {
		return false, err
	}
	defer f.Close()

	config := Config{}
	if err := json.NewDecoder(f).Decode(&config); err != nil {
		return false, err
	}

	return config.Server, nil
}

func main() {
	nagios := &gonagios.Nagios{}
	var consulConfig string
	var consulHost string

	flag.StringVar(&consulConfig, "config", "/etc/consul.d/main.json", "Consul config file to use")
	flag.StringVar(&consulHost, "consul", "localhost:8500", "Consul addr to check")
	flag.Parse()

	server, err := isServer(consulConfig)
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error checking if server: %v", err),
		)
	}
	if !server {
		nagios.ExitWithStatus(
			gonagios.OK,
			fmt.Sprintf("Host is not a consul server."),
		)
	}

	config := api.DefaultConfig()
	config.Address = consulHost
	client, err := api.NewClient(config)
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error creating consul client: %v", err),
		)
	}

	// Check the WAN status:
	datacenters, err := client.Catalog().Datacenters()
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error getting datacenters: %v", err),
		)
	}
	if len(datacenters) <= 1 {
		nagios.ExitWithStatus(
			gonagios.Critical,
			"Datacenter list is too short.",
		)
	}

	wanMembers, err := client.Agent().Members(true)
	if err != nil {
		nagios.ExitWithStatus(
			gonagios.Critical,
			fmt.Sprintf("Error getting wan members: %v", err),
		)
	}
	if len(wanMembers) <= 1 {
		nagios.ExitWithStatus(
			gonagios.Critical,
			"Wan Member list is too short.",
		)
	}

	status := gonagios.OK

	nagios.ExitWithStatus(status, "Checked peers, all online.",)
}


func nodePassing(client *api.Client, nodeName string) (bool, error) {
	checks, _, err := client.Health().Node(nodeName, nil)
	if err != nil {
		return false, err
	}
	for _, check := range checks {
		if check.CheckID == "serfHealth" && check.Status == "passing" {
			return true, nil
		}
	}
	return false, nil
}
