package main

import (
	"log"

	"time"

	"errors"

	"fmt"

	"code.justin.tv/release/courier/pkg/common"
	"code.justin.tv/release/courier/pkg/pack"
	"code.justin.tv/release/courier/pkg/ssh"
	"code.justin.tv/release/courier/pkg/structs"
	"code.justin.tv/release/courier/pkg/tar"
	"github.com/codegangsta/cli"
)

// courier deploy
func DeployCmd(c *cli.Context) error {
	startTime := time.Now()
	result := structs.COMMAND_FAILURE

	var options *structs.Options
	var d, r *structs.RemoteExecResult

	defer func() {
		totalFailed := 0

		if d != nil {
			totalFailed += d.FailCnt
		}

		if r != nil {
			totalFailed += r.FailCnt
		}

		if options == nil {
			options = &structs.Options{}
		}
		sendEventToSpade(c.String("spade-host"), "remote-deploy", false, totalFailed, startTime, time.Now(), result, c.App.Version, options)
	}()

	options, style, err := loadCourier(c)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	d, r, err = doDeploy(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{})
	printSummaryAll(d, r)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	if d.FailCnt > 0 {
		// some hosts failed to deploy
		return cli.NewExitError("Some hosts failed to deploy", ErrorExitCode)
	}

	result = structs.COMMAND_SUCCESS
	return nil
}

// courier rollback
func RollbackCmd(c *cli.Context) error {
	startTime := time.Now()
	result := structs.COMMAND_FAILURE
	var options *structs.Options
	var d, r *structs.RemoteExecResult

	defer func() {
		totalFailed := 0

		if d != nil {
			totalFailed += d.FailCnt
		}

		if r != nil {
			totalFailed += r.FailCnt
		}

		sendEventToSpade(c.String("spade-host"), "remote-rollback", false, totalFailed, startTime, time.Now(), result, c.App.Version, options)
	}()

	options, style, err := loadCourier(c)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	d, r, err = doRollback(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{})
	printSummaryAll(d, r)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	if d.FailCnt > 0 {
		// some hosts failed to roll back
		return cli.NewExitError("Some hosts failed to rollback", ErrorExitCode)
	}

	result = structs.COMMAND_SUCCESS
	return nil
}

func InstallCmd(c *cli.Context) {
	log.Fatal("not implemented yet")
}

func RestartCmd(c *cli.Context) {
	log.Fatal("not implemented yet")
}

func loadCourier(c *cli.Context) (*structs.Options, structs.Courier, error) {
	options := structs.NewOptions(c)

	err := options.LoadSkadiSettings()
	if err != nil {
		return nil, nil, err
	}

	if options.DeployConfig == nil {
		return nil, nil, errors.New("skadi deploy config not found")
	}

	var style structs.Courier
	artifact, err := options.DeployConfig.GetDeployArtifact(options.Environment)
	if err != nil {
		log.Fatalf("Error getting deploy artifact type: %v", err)
	}
	switch artifact {
	case "":
		fallthrough
	case "tar":
		log.Print("Using 'tar' settings.")
		style, err = tar.NewCourier(options)
		if err != nil {
			return nil, nil, err
		}
	case "pkg":
		// If deploy.json has "artifact": "pkg" then deploy using package based deploys.
		// See doc/states.md for more info.
		log.Print("Using 'pkg' settings.")
		style, err = pack.NewCourier(options)
		if err != nil {
			log.Fatal(err)
		}
	default:
		return nil, nil, fmt.Errorf("Unknown artifact setting: %v", artifact)
	}

	return options, style, nil
}

func printSummaryAll(d *structs.RemoteExecResult, r *structs.RemoteExecResult) {
	log.Printf("===================================================================")
	printSummary(d, "Distribution Summary")
	printSummary(r, "Restart Summary")
}

func printSummary(r *structs.RemoteExecResult, title string) {
	if r == nil {
		return
	}
	log.Printf("%v: %.2f%% succeeded (total/success/fail: %v/%v/%v)",
		title, 100-common.CalculateFailRate(r.NumHosts, r.FailCnt),
		r.NumHosts, r.NumHosts-r.FailCnt, r.FailCnt)
	if r.FailCnt > 0 {
		log.Printf("Failed hosts: %v", r.FailCnt)
		for _, h := range r.FailHosts {
			log.Printf("  %v - %v", h.Hostname, h.ErrString)
		}
	}
}
