package main

import (
	"fmt"
	"regexp"
	"strings"
	"testing"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/release/courier/pkg/structs"
	"code.justin.tv/release/courier/pkg/tar"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func getSampleOptions() structs.Options {
	return structs.Options{
		Hosts:        []string{"hostA", "hostB", "hostC"},
		Repo:         "testwith/sha",
		Environment:  "integration",
		DeployConfig: &api.DeployConfig{},

		// Setting the versions will avoid hitting Consul
		// FIXME: interface this
		KnownGoodVersion: "8c517a19abaf65bc7a4ee498bbd559173241d5af",
		DeployedVersion:  "e45cd377b94ccaede9fdd10b52cf369f06f7b8e8",
		//ConsulHost:  "api.us-west-2.prod.consul.live-video.a2z.com",
	}
}

type fakeCommandRunner struct{}

func (fcr fakeCommandRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	return nil
}

type fakeVersionUpdater struct {
	t   *testing.T
	sha string
}

func (cvu fakeVersionUpdater) UpdateDeployedVersion(options *structs.Options, goodVersion string) (err error) {
	require.Equal(cvu.t, cvu.sha, goodVersion)
	return nil
}

func assertTarInstallWithSha(t *testing.T, sha string, options *structs.Options, cmd string, flags map[string]*string, desc string) {
	require.Equal(t, sha, options.Sha, desc)
	re := regexp.MustCompile(` --target \d+-` + sha + "$")
	require.Regexp(t, re, cmd, "%s, got an invalid target", desc)
	require.Contains(t, flags, "sha")
	require.Equal(t, sha, *flags["sha"], "%s, got an invalid sha flag", desc)
}

func assertTarInstallWithoutSha(t *testing.T, sha string, options *structs.Options, cmd string, flags map[string]*string, desc string) {
	require.Equal(t, sha, options.DeployedVersion, desc)
	re := regexp.MustCompile(` --target \d+-` + sha + "$")
	require.Regexp(t, re, cmd, "%s, got an invalid target", desc)
	require.NotContains(t, flags, "sha")
}

func isLocalTarInstallCommand(cmd string) bool {
	return strings.HasPrefix(cmd, "sudo setuidgid jtv courier tar install")
}

func isLocalTarRestartCommand(cmd string) bool {
	return cmd == "sudo setuidgid jtv courier tar restart"
}

type tarDeployRunner struct {
	t               *testing.T
	deployedVersion string
}

func (fcr tarDeployRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarInstallCommand(cmd) {
		desc := fmt.Sprintf("while deploying host %s using sha=%s", host, fcr.deployedVersion)
		assertTarInstallWithoutSha(fcr.t, fcr.deployedVersion, options, cmd, flags, desc)
	} else if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestTarDeploy(t *testing.T) {
	options := getSampleOptions()
	commandRunner := tarDeployRunner{deployedVersion: options.DeployedVersion, t: t}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	distributionResult, restartResult, err := doDeploy(&options, style, commandRunner, fakeVersionUpdater{t, options.DeployedVersion})
	require.NoError(t, err)
	assert.Zero(t, distributionResult.FailCnt)
	assert.Equal(t, distributionResult.NumHosts, len(options.Hosts))
	assert.Zero(t, restartResult.FailCnt)
	assert.Equal(t, restartResult.NumHosts, len(options.Hosts))
}

type tarDeployWithShaRunner struct {
	sha string
	t   *testing.T
}

func (fcr tarDeployWithShaRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarInstallCommand(cmd) {
		desc := fmt.Sprintf("while deploying host %s using sha=%s", host, fcr.sha)
		assertTarInstallWithSha(fcr.t, fcr.sha, options, cmd, flags, desc)
	} else if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestTarDeployWithSha(t *testing.T) {
	sha := "f0c12e9bc1436be5405289f00dfed94d1f964ce7"
	commandRunner := tarDeployWithShaRunner{sha: sha, t: t}
	options := getSampleOptions()
	options.Sha = sha
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	options.Sha = sha
	distributionResult, restartResult, err := doDeploy(&options, style, commandRunner, fakeVersionUpdater{t, sha})
	require.NoError(t, err)
	assert.Zero(t, distributionResult.FailCnt)
	assert.Equal(t, distributionResult.NumHosts, len(options.Hosts))
	assert.Zero(t, restartResult.FailCnt)
	assert.Equal(t, restartResult.NumHosts, len(options.Hosts))
}

type rollbackSuccessRunner struct {
	t               *testing.T
	goodVersion     string
	deployedVersion string
}

func (fcr rollbackSuccessRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarInstallCommand(cmd) {
		desc := fmt.Sprintf("while rolling back host %s to good version=%s", host, fcr.goodVersion)
		assertTarInstallWithSha(fcr.t, fcr.goodVersion, options, cmd, flags, desc)
	} else if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestRollbackSuccess(t *testing.T) {
	options := getSampleOptions()
	commandRunner := rollbackSuccessRunner{t: t, goodVersion: options.KnownGoodVersion, deployedVersion: options.DeployedVersion}
	dvu := fakeVersionUpdater{t: t, sha: options.KnownGoodVersion}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	distributionResult, restartResult, err := doRollback(&options, style, commandRunner, dvu)
	require.NoError(t, err)
	assert.Zero(t, distributionResult.FailCnt)
	assert.Equal(t, distributionResult.NumHosts, len(options.Hosts))
	assert.Zero(t, restartResult.FailCnt)
	assert.Equal(t, restartResult.NumHosts, len(options.Hosts))
}

func TestRollbackToSameVersionFailure(t *testing.T) {
	options := getSampleOptions()
	options.DeployedVersion = options.KnownGoodVersion
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	distributionResult, restartResult, err := doRollback(&options, style, fakeCommandRunner{}, fakeVersionUpdater{})
	require.Error(t, err)
	assert.Nil(t, distributionResult)
	assert.Nil(t, restartResult)
}

type restartSuccessRunner struct {
	t               *testing.T
	goodVersion     string
	deployedVersion string
}

func (fcr restartSuccessRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestRestartSuccess(t *testing.T) {
	options := getSampleOptions()
	options.Hosts = []string{"host1"}
	commandRunner := restartSuccessRunner{t: t, goodVersion: options.KnownGoodVersion, deployedVersion: options.DeployedVersion}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	remoteExecResult, err := doRestart(&options, style, commandRunner, fakeVersionUpdater{t: t})
	require.NoError(t, err)
	assert.Zero(t, remoteExecResult.FailCnt)
	assert.Equal(t, remoteExecResult.NumHosts, len(options.Hosts))
}

type restartFailureTriggersRollbackRunner struct {
	t               *testing.T
	goodVersion     string
	deployedVersion string
	gotFirstRestart bool
}

func (fcr restartFailureTriggersRollbackRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarRestartCommand(cmd) {
		// we only want to make the first restart fail, not the restart after the rollback
		if !fcr.gotFirstRestart {
			fcr.gotFirstRestart = true
			return fmt.Errorf("forcing an error to trigger a rollback during restart")
		}
	} else if isLocalTarInstallCommand(cmd) {
		desc := fmt.Sprintf("while rolling back host %s to good version=%s after a failed restart", host, fcr.goodVersion)
		assertTarInstallWithSha(fcr.t, fcr.goodVersion, options, cmd, flags, desc)
	} else {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestRestartFailureTriggersRollback(t *testing.T) {
	options := getSampleOptions()
	options.Hosts = []string{"host1"}
	commandRunner := restartFailureTriggersRollbackRunner{t: t, goodVersion: options.KnownGoodVersion, deployedVersion: options.DeployedVersion, gotFirstRestart: false}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	remoteExecResult, err := doRestart(&options, style, commandRunner, fakeVersionUpdater{t, options.KnownGoodVersion})
	require.Error(t, err)
	assert.Equal(t, remoteExecResult.FailCnt, len(options.Hosts))
}

type distributeSuccessRunner struct {
	t               *testing.T
	goodVersion     string
	deployedVersion string
}

func (fcr distributeSuccessRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarInstallCommand(cmd) {
		desc := fmt.Sprintf("while distributing version %s in host %s", fcr.deployedVersion, host)
		assertTarInstallWithoutSha(fcr.t, fcr.deployedVersion, options, cmd, flags, desc)
	} else if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestDistributeSuccess(t *testing.T) {
	options := getSampleOptions()
	options.Hosts = []string{"host1"}
	commandRunner := distributeSuccessRunner{t: t, goodVersion: options.KnownGoodVersion, deployedVersion: options.DeployedVersion}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	remoteExecResult, err := doDistribute(&options, style, commandRunner, fakeVersionUpdater{t, options.KnownGoodVersion})
	require.NoError(t, err)
	assert.Zero(t, remoteExecResult.FailCnt)
	require.Equal(t, remoteExecResult.NumHosts, len(options.Hosts))
}

type distributeFailureTriggersRollbackRunner struct {
	t                    *testing.T
	goodVersion          string
	deployedVersion      string
	gotFirstDistribution bool
}

func (fcr distributeFailureTriggersRollbackRunner) RunRemoteHost(host string, cmd string, options *structs.Options, flags map[string]*string) error {
	if isLocalTarInstallCommand(cmd) {
		// we only want to make the first restart fail, not the restart after the rollback
		if !fcr.gotFirstDistribution {
			fcr.gotFirstDistribution = true
			return fmt.Errorf("forcing an error to trigger a rollback during distribute")
		} else {
			desc := fmt.Sprintf("while rolling back host %s to good version=%s after a failed distribute", host, fcr.goodVersion)
			assertTarInstallWithSha(fcr.t, fcr.goodVersion, options, cmd, flags, desc)
		}
	} else if !isLocalTarRestartCommand(cmd) {
		require.FailNow(fcr.t, "unhandled command: %s", cmd)
	}
	return nil
}

func TestDistributeFailureTriggersRollback(t *testing.T) {
	options := getSampleOptions()
	options.Hosts = []string{"host1"}
	commandRunner := distributeFailureTriggersRollbackRunner{t: t, goodVersion: options.KnownGoodVersion, deployedVersion: options.DeployedVersion, gotFirstDistribution: false}
	style, err := tar.NewCourier(&options)
	require.NoError(t, err)
	remoteExecResult, err := doDistribute(&options, style, commandRunner, fakeVersionUpdater{t, options.KnownGoodVersion})
	require.Error(t, err)
	assert.Equal(t, remoteExecResult.FailCnt, len(options.Hosts))
}
