package main

import (
	"fmt"
	"log"

	"time"

	"code.justin.tv/dta/rtevent"
	"code.justin.tv/release/courier/pkg/pack"
	"code.justin.tv/release/courier/pkg/ssh"
	"code.justin.tv/release/courier/pkg/structs"
	"code.justin.tv/release/courier/pkg/tar"
	"github.com/codegangsta/cli"
)

func deprecated() {
	log.Print(`This courier command is deprecated!! Please use "courier deploy" for remote usage or "courier install|restart" for local usage.`)
}

// courier tar status
func LocalTarStatusCmd(c *cli.Context) error {
	var err error
	var startTime = time.Now()
	result := structs.COMMAND_FAILURE
	options := structs.NewOptions(c)

	defer func() {
		sendEventToSpade(c.String("spade-host"), "local-tar-status", true, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	sha, err := options.GetDeployedVersion()
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	options.Sha = sha
	targetName, err := options.GetTargetName()
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	installedVersion, err := style.LocalStatus(
		c.String("repo"),
		c.String("environment"),
		c.String("dir"),
		sha,
		targetName,
	)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	log.Printf("Current installed version: %s\n", installedVersion)
	result = structs.COMMAND_SUCCESS
	return nil
}

// courier tar install
func LocalTarInstallCmd(c *cli.Context) error {
	var err error
	options := structs.NewOptions(c)
	success := false
	noop := false
	startTime := time.Now()

	defer func() {
		if rtclient, err := rtevent.NewPublisherClient(); err == nil {
			event := &rtevent.DeployEvent{
				App:         options.Repo,
				Sha:         options.Sha,
				Environment: options.Environment,
				Success:     success,
				Phase:       "install-local",
				Deployer:    "courier",
			}
			if success == false && err != nil {
				event.Desc = fmt.Sprintf("%v", err)
			}
			err = rtclient.SendDeployEvent(event)
		} else {
			log.Println(err)
		}
	}()

	defer func() {
		result := structs.COMMAND_FAILURE
		if success {
			if noop {
				result = structs.COMMAND_NOOP
			} else {
				result = structs.COMMAND_SUCCESS
			}
		}
		sendEventToSpade(c.String("spade-host"), "local-tar-install", true, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	var sha = options.Sha
	if sha == "" {
		sha, err = options.GetDeployedVersion()
		if err != nil {
			return cli.NewExitError(err, ErrorExitCode)
		}
		options.Sha = sha
	}

	log.Printf("Deploying version with SHA: %s", sha)

	targetName, err := options.GetTargetName()
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)

	}
	noop, err = style.LocalInstall(
		c.String("repo"),
		c.String("environment"),
		c.String("dir"),
		sha,
		targetName,
		c.String("config-path"),
		c.Bool("skip-symlink"),
		c.Bool("symlink-in-restart"),
	)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	success = true
	return nil
}

// courier tar restart
func LocalTarRestartCmd(c *cli.Context) error {
	var err error
	options := structs.NewOptions(c)
	success := false
	startTime := time.Now()
	defer func() {
		if rtclient, err := rtevent.NewPublisherClient(); err == nil {
			event := &rtevent.DeployEvent{
				App:         options.Repo,
				Sha:         options.Sha,
				Environment: options.Environment,
				Success:     success,
				Phase:       "restart-local",
				Deployer:    "courier",
			}
			if success == false && err != nil {
				event.Desc = fmt.Sprintf("%v", err)
			}
			err = rtclient.SendDeployEvent(event)
		} else {
			log.Println(err)
		}
	}()

	defer func() {
		result := structs.COMMAND_FAILURE
		if success {
			result = structs.COMMAND_SUCCESS
		}
		sendEventToSpade(c.String("spade-host"), "local-tar-restart", true, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	var sha = options.Sha
	if sha == "" {
		sha, err = options.GetDeployedVersion()
		if err != nil {
			return cli.NewExitError(err, ErrorExitCode)
		}
		options.Sha = sha
	}

	log.Printf("Deploying version: %s", sha)

	targetName, err := options.GetTargetName()
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	if err := options.LoadSkadiSettings(); err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	err = style.LocalRestart(
		options,
		c.String("repo"),
		c.String("environment"),
		c.String("dir"),
		sha,
		targetName,
		c.Bool("symlink-in-restart"),
	)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}
	success = true
	return nil
}

// courier tar deploy
func TarDeployCmd(c *cli.Context) error {
	deprecated()

	startTime := time.Now()
	options := structs.NewOptions(c)
	result := structs.COMMAND_FAILURE

	defer func() {
		sendEventToSpade(c.String("spade-host"), "tar-deploy", false, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	if _, _, err := doDeploy(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{}); err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	result = structs.COMMAND_SUCCESS
	return nil
}

// courier tar deploy install
func TarDeployInstallCmd(c *cli.Context) error {
	deprecated()

	startTime := time.Now()
	result := structs.COMMAND_FAILURE
	options := structs.NewOptions(c)

	defer func() {
		sendEventToSpade(c.String("spade-host"), "tar-deploy-install", false, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	if _, err := doDistribute(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{}); err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	result = structs.COMMAND_SUCCESS
	return nil
}

// courier tar deploy restart
func TarDeployRestartCmd(c *cli.Context) error {
	deprecated()

	startTime := time.Now()
	success := false
	options := structs.NewOptions(c)

	defer func() {
		result := structs.COMMAND_FAILURE
		if success {
			result = structs.COMMAND_SUCCESS
		}
		sendEventToSpade(c.String("spade-host"), "tar-deploy-restart", false, 0, startTime, time.Now(), result, c.App.Version, options)
	}()

	style, err := tar.NewCourier(options)
	if err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	if _, err = doRestart(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{}); err != nil {
		return cli.NewExitError(err, ErrorExitCode)
	}

	return nil
}

// courier pkg install
func LocalPkgInstallCmd(c *cli.Context) {
	options := structs.NewOptions(c)
	success := false
	defer func() {
		if rtclient, err := rtevent.NewPublisherClient(); err == nil {
			event := &rtevent.DeployEvent{
				App:         options.Repo,
				Sha:         options.Sha,
				Environment: options.Environment,
				Success:     success,
				Phase:       "install-local",
				Deployer:    "courier",
			}
			if success == false && err != nil {
				event.Desc = fmt.Sprintf("%v", err)
			}
			err = rtclient.SendDeployEvent(event)
		} else {
			log.Println(err)
		}
	}()

	var err error
	var sha = options.Sha
	if sha == "" {
		sha, err = options.GetDeployedVersion()
		if err != nil {
			log.Fatal(err, sha)
		}
		options.Sha = sha
	}

	log.Printf("Deploying version with SHA: %s", sha)

	style, err := pack.NewCourier(options)
	if err != nil {
		log.Fatal(err)
	}
	err = style.LocalInstall(
		c.String("repo"),
		sha,
	)
	if err != nil {
		log.Fatal(err)
	}
	success = true
}

// courier pkg remove
func LocalPkgRemoveCmd(c *cli.Context) {
	options := structs.NewOptions(c)
	var err error
	log.Printf("Removing %s", c.String("repo"))

	style, err := pack.NewCourier(options)
	if err != nil {
		log.Fatal(err)
	}
	err = style.LocalRemove(
		c.String("repo"),
	)
	if err != nil {
		log.Fatal(err)
	}
}

// courier pkg deploy
func PkgDeployCmd(c *cli.Context) {
	deprecated()

	options := structs.NewOptions(c)

	style, err := pack.NewCourier(options)
	if err != nil {
		log.Fatal(err)
	}

	distributionResult, err := doDistribute(options, style, ssh.SshCommandRunner{}, structs.ConsulVersionUpdater{})
	if err != nil {
		log.Fatalf("Deploy result: %v: %v", err, distributionResult)
	}
}
