package main

import (
	"context"
	"fmt"
	"log"

	"net/url"

	"time"

	"os"

	"code.justin.tv/common/spade-client-go/spade"
	"code.justin.tv/release/courier/pkg/distribution"
	"code.justin.tv/release/courier/pkg/restart"
	"code.justin.tv/release/courier/pkg/structs"
	"github.com/codegangsta/cli"
)

var AppVersion string = "dev"

const SpadeEventName = "devtools_courier_usage"
const ErrorExitCode = 1

func main() {

	app := cli.NewApp()
	app.Version = AppVersion
	app.Usage = "Fetches and installs twitch-written applications."

	// shared flags
	spadeFlag := cli.StringFlag{
		Name:  "spade-host",
		Value: "spade.internal.di.twitch.a2z.com",
	}
	consulFlag := cli.StringFlag{
		Name:  "consul-host",
		Value: "localhost:8500",
	}
	consulFallbackFlag := cli.StringFlag{
		Name:  "consul-fallback-dc",
		Value: "wanmaster",
	}
	skadiFlag := cli.StringFlag{
		Name:  "skadi-host",
		Value: "https://deploy.xarth.tv",
	}
	hostsFlag := cli.StringFlag{
		Name:  "hosts",
		Usage: "comma-separated list of target hosts",
	}
	envFlag := cli.StringFlag{
		Name:   "environment",
		Usage:  "environment to use when looking up app version in consul",
		EnvVar: "ENVIRONMENT",
	}
	repoFlag := cli.StringFlag{
		Name:  "repo",
		Usage: "repository fullname. Format: owner/repo",
	}
	shortCircuitFlag := cli.BoolFlag{
		Name:  "short-circuit",
		Usage: "Fail the deploy if any remote nodes return an error",
	}
	skipRestartFlag := cli.BoolFlag{
		Name:  "skip-restart",
		Usage: "Skip restarting the application",
	}
	skipSymlinkFlag := cli.BoolFlag{
		Name:  "skip-symlink",
		Usage: "Do not update the current symlink in the install phase",
	}
	symlinkInRestartFlag := cli.BoolFlag{
		Name:  "symlink-in-restart",
		Usage: "Update the current symlink in the restart phase (this also enables skip-symlink)",
	}
	retriesFlag := cli.IntFlag{
		Name:  "retries",
		Usage: "Number of times we try to performan an operation on remote nodes (default: 3)",
		Value: 3,
	}
	userFlag := cli.StringFlag{
		Name:  "user",
		Usage: "The username we will run code under (default: jtv)",
		Value: "",
	}

	// tar flags
	dirFlag := cli.StringFlag{
		Name:  "dir",
		Usage: "directory containing releases and current symlink (tar only)",
	}
	targetFlag := cli.StringFlag{
		Name:  "target",
		Usage: "name to use for the target name in the releases dir (tar only)",
	}
	shaFlag := cli.StringFlag{
		Name:  "sha",
		Usage: "override the sha to deploy rather than querying latest from consul (tar+pkg only)",
	}
	rollbackShaFlag := cli.StringFlag{
		Name:  "rollback-sha",
		Usage: "override the sha to rollback rather than querying consul (tar+pkg only)",
	}
	configFlag := cli.StringFlag{
		Name:  "config-path",
		Usage: "override the default courier.conf (/etc/courier.conf) filepath",
		Value: "/etc/courier.conf",
	}
	sshDialTimeoutFlag := cli.IntFlag{
		Name:  "ssh-dial-timeout",
		Usage: "SSH dial connection timeout in seconds (default: 20)",
		Value: 20,
	}

	allFlags := []cli.Flag{spadeFlag, consulFlag, consulFallbackFlag, skadiFlag, hostsFlag, envFlag, repoFlag, shortCircuitFlag, skipSymlinkFlag, symlinkInRestartFlag, skipRestartFlag, dirFlag, targetFlag, rollbackShaFlag, shaFlag, configFlag, retriesFlag, userFlag, sshDialTimeoutFlag}

	pkgFlags := []cli.Flag{consulFlag, skadiFlag, hostsFlag, envFlag, repoFlag, shortCircuitFlag, rollbackShaFlag, shaFlag, retriesFlag, userFlag, sshDialTimeoutFlag}

	app.Commands = []cli.Command{
		{
			Name:   "deploy",
			Usage:  "(Remote) Distribute and restart the application",
			Action: DeployCmd,
			Flags:  allFlags,
		},
		{
			Name:   "rollback",
			Usage:  "(Remote) Rollback the application to the last known good version.",
			Action: RollbackCmd,
			Flags:  allFlags,
		},
		{
			Name:   "install",
			Usage:  "(Local) Install the application on the local machine.",
			Action: InstallCmd,
			Flags:  allFlags,
		},
		{
			Name:   "restart",
			Usage:  "(Local) Restart the currently installed application.",
			Action: RestartCmd,
			Flags:  allFlags,
		},
		{
			Name:  "tar",
			Usage: "Install and run tar-packaged applications",
			Subcommands: []cli.Command{
				{
					Name:   "install",
					Usage:  "(Local) Install tar to DIR/releases/SHA and point to it with 'current' symlink",
					Action: LocalTarInstallCmd,
					Flags:  allFlags,
				},
				{
					Name:   "restart",
					Usage:  "(Local) Restart the installed application.",
					Action: LocalTarRestartCmd,
					Flags:  allFlags,
				},
				{
					Name:   "status",
					Usage:  "(Local) Return the currently installed version of the application if it's up to date",
					Action: LocalTarStatusCmd,
					Flags:  allFlags,
				},
				{
					Name:   "deploy",
					Usage:  "(Remote) Install and restart the current image on the given hosts",
					Action: TarDeployCmd,
					Flags:  allFlags,
					Subcommands: []cli.Command{
						{
							Name:   "install",
							Usage:  "(Remote) Install tar to DIR/releases/SHA and point to it with 'current' symlink",
							Action: TarDeployInstallCmd,
							Flags:  allFlags,
						},
						{
							Name:   "restart",
							Usage:  "(Remote) Restart the installed application",
							Action: TarDeployRestartCmd,
							Flags:  allFlags,
						},
					},
				},
			},
		},
		{
			Name:  "pkg",
			Usage: "Install and run deb/rpm-packaged applications. Service restarts are handled by the package",
			Subcommands: []cli.Command{
				{
					Name:   "install",
					Usage:  "(Local) Install package",
					Action: LocalPkgInstallCmd,
					Flags:  pkgFlags,
				},
				{
					Name:   "deploy",
					Usage:  "(Remote) Install the current sha on the given hosts",
					Action: PkgDeployCmd,
					Flags:  pkgFlags,
				},
				{
					Name:   "remove",
					Usage:  "(local) Remove the service. Provided primarily for integration tests",
					Action: LocalPkgRemoveCmd,
					Flags:  pkgFlags,
				},
			},
		},
	}

	app.RunAndExitOnError()
}

func sendEventToSpade(spadeHost string, command string, localRun bool, numRemoteFailedHosts int, commandStartTime time.Time, commandEndTime time.Time, commandResult structs.CommandResult, version string, options *structs.Options) {
	hostname, err := os.Hostname()
	if err != nil {
		log.Println(err)
		return
	}

	courierEvent := structs.NewCourierEvent(command, hostname, localRun, numRemoteFailedHosts, commandStartTime, commandEndTime, commandResult, version, options)

	spadeClient, err := spade.NewClient(spade.InitBaseURL(url.URL{Scheme: "https", Host: spadeHost, Path: "/track"}))
	if err != nil {
		log.Println(err)
		return
	}

	ctx, cancel := context.WithTimeout(context.Background(), 2*time.Second)
	defer cancel()
	err = spadeClient.TrackEvent(ctx, SpadeEventName, courierEvent)
	if err != nil {
		log.Println(err)
		return
	}
}

func doDeploy(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner, dvu structs.DeployedVersionUpdater) (*structs.RemoteExecResult, *structs.RemoteExecResult, error) {
	distributionResult, err := doDistribute(options, style, commandRunner, dvu)
	if err != nil {
		return distributionResult, nil, err
	}

	// Remove failed hosts from the target
	FailedHostList := make([]string, len(distributionResult.FailHosts))
	for _, host := range distributionResult.FailHosts {
		FailedHostList = append(FailedHostList, host.Hostname)
	}
	options.RemoveHosts(FailedHostList)

	restartResult, err := doRestart(options, style, commandRunner, dvu)
	if err != nil {
		return distributionResult, restartResult, err
	}

	return distributionResult, restartResult, nil
}

func doRollback(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner, dvu structs.DeployedVersionUpdater) (*structs.RemoteExecResult, *structs.RemoteExecResult, error) {
	goodVersion, err := rollbackVersion(options, dvu)
	if err != nil {
		return nil, nil, fmt.Errorf("unable to rollback version to previous known good version! %v", err)
	}
	options.Sha = goodVersion
	if len(options.Hosts) == 0 {
		return nil, nil, fmt.Errorf("--hosts ($HOSTS) can't be blank")
	}
	if options.Repo == "" {
		return nil, nil, fmt.Errorf("--repo ($REPO) must be in format: '<owner>/<repo>'")
	}
	if options.SkipRestart {
		log.Println("Skipping restart due to --skip-restart")
		return nil, nil, nil
	}

	if err := options.LoadSkadiSettings(); err != nil {
		return nil, nil, err
	}

	distributionResult, distributionErr := distribution.Execute(options, style, commandRunner)
	if distributionErr != nil {
		return distributionResult, nil, fmt.Errorf("Fail to rollback back to last known good version during distribution: %v", distributionErr)
	}

	// Remove failed hosts from the target
	FailedHostList := make([]string, len(distributionResult.FailHosts))
	for _, host := range distributionResult.FailHosts {
		FailedHostList = append(FailedHostList, host.Hostname)
	}
	options.RemoveHosts(FailedHostList)

	restartResult, restartErr := restart.Execute(options, style, commandRunner)
	if restartErr != nil {
		return distributionResult, restartResult, fmt.Errorf("Fail to rollback back to last known good version during restart: %v", restartErr)
	}

	return distributionResult, restartResult, nil
}

func doDistribute(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner, dvu structs.DeployedVersionUpdater) (*structs.RemoteExecResult, error) {
	if len(options.Hosts) == 0 {
		return nil, fmt.Errorf("--hosts ($HOSTS) can't be blank")
	}
	if options.Repo == "" {
		return nil, fmt.Errorf("--repo ($REPO) must be in format: '<owner>/<repo>'")
	}

	if err := options.LoadSkadiSettings(); err != nil {
		return nil, err
	}

	distributionResult, originalErr := distribution.Execute(options, style, commandRunner)
	if originalErr != nil {
		log.Printf("Distribution failed, rolling back to last known good version: %v", originalErr)
		goodVersion, err := rollbackVersion(options, dvu)
		if err != nil {
			return distributionResult, fmt.Errorf("unable to rollback version to previous known good version! %v", err)
		}
		options.Sha = goodVersion
		if _, err := distribution.Execute(options, style, commandRunner); err != nil {
			return distributionResult, err
		}

		return distributionResult, originalErr
	}

	return distributionResult, nil
}

func doRestart(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner, dvu structs.DeployedVersionUpdater) (*structs.RemoteExecResult, error) {
	if options.SkipRestart {
		log.Println("Skipping restart due to --skip-restart")
		return nil, nil
	}

	if len(options.Hosts) == 0 {
		return nil, fmt.Errorf("--hosts ($HOSTS) can't be blank")
	}

	if err := options.LoadSkadiSettings(); err != nil {
		return nil, err
	}

	restartResult, originalErr := restart.Execute(options, style, commandRunner)
	if originalErr != nil {
		log.Printf("Restart failed, rolling back to last known good version: %v", originalErr)

		goodVersion, err := rollbackVersion(options, dvu)
		if err != nil {
			return restartResult, fmt.Errorf("unable to rollback version to previous known good version! %v", err)
		}
		options.Sha = goodVersion
		if _, err := distribution.Execute(options, style, commandRunner); err != nil {
			return restartResult, err
		}

		if _, err := restart.Execute(options, style, commandRunner); err != nil {
			return restartResult, err
		}

		return restartResult, originalErr
	}

	return restartResult, nil
}

func rollbackVersion(options *structs.Options, dvu structs.DeployedVersionUpdater) (goodVersion string, err error) {
	goodVersion, err = options.GetKnownGoodVersion()
	if err != nil {
		return
	}

	deployedVersion, err := options.GetDeployedVersion()
	if err != nil {
		return
	}

	if deployedVersion == goodVersion {
		err = fmt.Errorf("deploy version and last known good version are the same, rollback failed")
		return
	}

	err = dvu.UpdateDeployedVersion(options, goodVersion)
	return
}
