package artifact

import (
	"errors"
	"fmt"
	"io"
	"net"
	"net/http"
	"net/url"
	"os"
	"time"

	"code.justin.tv/release/courier/pkg/config"
)

var (
	client *http.Client
)

func init() {
	client = &http.Client{
		Transport: &http.Transport{
			Dial: (&net.Dialer{
				Timeout:   30 * time.Second,
				KeepAlive: 30 * time.Second,
				DualStack: true,
			}).Dial,
			TLSHandshakeTimeout:   10 * time.Second,
			ExpectContinueTimeout: 1 * time.Second,
		},
	}
}

// HTTPDownloader is an artifact Downloader that downloads artifacts from BaseURL.
type HTTPDownloader struct {
	BaseURL         *url.URL
	MaxTransferRate int
}

// LoadHTTPDownloader creates a new HTTPDownloader from config values in configJson. It requires
// http_baseurl be set.
func LoadHTTPDownloader(configJSON config.Config) (*HTTPDownloader, error) {
	d := new(HTTPDownloader)
	baseurl, err := configJSON.StringValue("http_baseurl")
	if err != nil {
		return nil, err
	}
	if baseurl == "" {
		return nil, errors.New("http_baseurl must be set")
	}
	if d.BaseURL, err = url.Parse(baseurl); err != nil {
		return nil, fmt.Errorf("Could not parse http_baseurl: %v", err)
	}
	if d.MaxTransferRate, err = configJSON.IntValue("max_transfer_rate"); err != nil {
		return nil, err
	}
	return d, nil
}

// DownloadArtifact downloads the artifact at repo and sha to f.
func (d *HTTPDownloader) DownloadArtifact(f *os.File, repo, sha string) (int, error) {
	url, err := d.BaseURL.Parse(fmt.Sprintf("%v/%v.tgz", repo, sha))
	if err != nil {
		return 0, err
	}
	resp, err := http.Get(url.String())
	if err != nil {
		return 0, err
	}
	if err != nil {
		return 0, err
	}
	if resp.StatusCode >= http.StatusBadRequest {
		resp.Body.Close()
		return 0, errors.New(resp.Status)
	}
	var reader io.ReadCloser = resp.Body
	if d.MaxTransferRate > 0 {
		reader = newRateLimitReader(d.MaxTransferRate, reader)
	}
	defer reader.Close()
	nr, err := io.Copy(f, reader)
	if err != nil {
		return int(nr), err
	}
	return int(nr), nil
}
