package artifact

import (
	"bytes"
	"sync"
	"testing"
	"time"
)

type ClosingBuffer struct {
	*bytes.Buffer
}

func (cb *ClosingBuffer) Close() error {
	return nil
}

// Checks that the rateLimitReader tokens increase over time
func TestRateLimitReaderTokenIncrease(t *testing.T) {
	r := newRateLimitReader(1000, nil)
	t1 := r.tokens
	time.Sleep(1 * time.Second)
	t2 := r.tokens
	if t2 <= t1 {
		t.Errorf("rateLimitReader tokens did not increase over time")
	}
}

// Checks that the rateLimitReader tokens stop increasing at maximum
func TestRateLimitReaderMaxTokens(t *testing.T) {
	r := newRateLimitReader(1000, nil)
	t1 := 1000 * rateLimitBufferFactor
	r.tokens = t1
	time.Sleep(1 * time.Second)
	t2 := r.tokens
	if t2 > t1 {
		t.Errorf("rateLimitReader tokens increased past maximum")
	}
}

// Checks that the rateLimitReader reads 0 bytes when there are no tokens
func TestRateLimitReaderReadNoTokens(t *testing.T) {
	r := &rateLimitReader{
		wr: &ClosingBuffer{
			Buffer: bytes.NewBufferString("test"),
		},
		tokens:      0,
		tokensMutex: new(sync.Mutex),
	}
	nr, err := r.Read(nil)
	if err != nil {
		t.Fatal(err)
	}
	if nr > 0 {
		t.Errorf("rateLimitReader read bytes when reader had 0 tokens")
	}
	if r.tokens < 0 {
		t.Errorf("rateLimitReader tokens are <0")
	}
}

// Checks that the rateLimitReader reads x bytes when there are x tokens < len(b)
func TestRateLimitReaderReadOnlyXBytes(t *testing.T) {
	var tokens = 2
	r := &rateLimitReader{
		wr: &ClosingBuffer{
			Buffer: bytes.NewBufferString("test"),
		},
		tokens:      tokens,
		tokensMutex: new(sync.Mutex),
	}
	b := make([]byte, 4)
	nr, err := r.Read(b)
	if err != nil {
		t.Fatal(err)
	}
	if nr > tokens {
		t.Errorf("rateLimitReader read bytes > tokens")
	}
	if b[tokens] != 0 {
		t.Errorf("rateLimitReader wrote into b past tokens index")
	}
	if r.tokens > 0 {
		t.Errorf("rateLimitReader did not properly decrement tokens")
	}
}

// Checks that the rateLimitReader reads all bytes when there are x tokens > len(b)
func TestRateLimitReaderReadAllBytes(t *testing.T) {
	var tokens = 5
	var testString = "test"
	r := &rateLimitReader{
		wr: &ClosingBuffer{
			Buffer: bytes.NewBufferString(testString),
		},
		tokens:      tokens,
		tokensMutex: new(sync.Mutex),
	}
	b := make([]byte, 3)
	nr, err := r.Read(b)
	if err != nil {
		t.Fatal(err)
	}
	if nr != len(b) {
		t.Errorf("rateLimitReader read bytes != len(b)")
	}
	if string(b) != testString[0:len(b)] {
		t.Errorf("rateLimitReader did not read correctly into b")
	}
	if r.tokens > tokens-len(b) {
		t.Errorf("rateLimitReader did not properly decrement tokens")
	}
}

// Checks that the rateLimitWriterAt tokens increase over time
func TestRateLimitWriterAtTokenIncrease(t *testing.T) {
	w := newRateLimitWriterAt(1000, nil)
	t1 := w.tokens
	time.Sleep(1 * time.Second)
	t2 := w.tokens
	if t2 <= t1 {
		t.Errorf("rateLimitWriterAt tokens did not increase over time")
	}
}

// Checks that the rateLimitWriterAt tokens stop increasing at maximum
func TestRateLimitWriterAtMaxTokens(t *testing.T) {
	w := newRateLimitWriterAt(1000, nil)
	t1 := 1000 * rateLimitBufferFactor
	w.tokens = t1
	time.Sleep(1 * time.Second)
	t2 := w.tokens
	if t2 > t1 {
		t.Errorf("rateLimitWriterAt tokens increased past maximum")
	}
}

// Checks that the writer returns from waiting when there are enough tokens
func TestRateLimitWriterWaitForTokens(t *testing.T) {
	w := newRateLimitWriterAt(1000, nil)
	w.tokens = 1000
	wc := make(chan struct{})
	go func() {
		w.waitForTokens(1000)
		wc <- struct{}{}
	}()

	select {
	case <-wc:
		return
	case <-time.After(time.Duration(1) * time.Second):
		t.Fatal("waitForTokens took longer than 1 second")
	}
}

// Checks that the write queue is cleared when the rateLimitWriterAt is closed
func TestRateLimitWriterCloseClearQueue(t *testing.T) {
	w := &rateLimitWriterAt{
		queueMutex: new(sync.Mutex),
		ticker:     time.NewTicker(time.Second / rateLimitTickRate),
	}
	queueEntry1 := &writeQueueEntry{
		capacity: 1000,
		wc:       make(chan struct{}),
	}
	// send through intermediate channel to avoid blocking on send
	wc := make(chan struct{})
	go func() {
		s := <-queueEntry1.wc
		wc <- s
	}()
	w.writeQueue = append(w.writeQueue, queueEntry1)
	time.Sleep(time.Duration(20) * time.Millisecond)
	w.Close()
	select {
	case <-wc:
		return
	case <-time.After(time.Duration(100) * time.Second):
		t.Fatal("queue entry was not cleared")
	}
}
