package artifact

import (
	"context"
	"fmt"
	"io"
	"os"
	"time"

	"code.justin.tv/release/courier/pkg/config"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
)

const (
	defaultS3Bucket   = "devtools-deploy-artifacts"
	defaultS3Region   = "us-west-2"
	defaultRetryLimit = -1
	defaultTimeout    = 60 * 5
)

// S3Downloader is an artifact Downloader that downloads artifacts from Bucket.
type S3Downloader struct {
	Bucket          string
	MaxTransferRate int
	Concurrency     int
	Timeout         time.Duration
	S3              s3iface.S3API
}

// LoadS3Downloader creates a new S3Downloader from config values in configJson. It requires
// s3_bucket be set. If s3_region is set, it will be used, otherwise the default region is
// us-west-2. If access_key and secret_key are set, they will be used, otherwise courier will
// attempt to get authorization credentials from environment variables, an instance profile, or an
// AWS ini profile.
func LoadS3Downloader(configJSON config.Config) (*S3Downloader, error) {
	d := new(S3Downloader)
	var (
		regionName, accessKey, secretKey string
		retryLimit, timeout              int
		err                              error
	)
	if d.Bucket, err = configJSON.StringValueOrDefault("s3_bucket", defaultS3Bucket); err != nil {
		return nil, fmt.Errorf("Could not read s3_bucket: %v", err)
	}
	if d.Concurrency, err = configJSON.IntValue("s3_concurrency"); err != nil {
		return nil, err
	}
	if d.MaxTransferRate, err = configJSON.IntValue("max_transfer_rate"); err != nil {
		return nil, err
	}
	if regionName, err = configJSON.StringValueOrDefault("s3_region", defaultS3Region); err != nil {
		return nil, fmt.Errorf("Could not read s3_region: %v", err)
	}
	if retryLimit, err = configJSON.IntValueOrDefault("s3_retry_limit", defaultRetryLimit); err != nil {
		return nil, err
	}
	if timeout, err = configJSON.IntValueOrDefault("timeout", defaultTimeout); err != nil {
		return nil, err
	}
	d.Timeout = time.Duration(timeout) * time.Second
	accessKey, _ = configJSON.StringValue("access_key")
	secretKey, _ = configJSON.StringValue("secret_key")
	var creds *credentials.Credentials
	if accessKey != "" && secretKey != "" {
		creds = credentials.NewStaticCredentials(accessKey, secretKey, "")
	}
	sess := session.Must(session.NewSession(&aws.Config{
		Credentials: creds,
		MaxRetries:  &retryLimit,
		Region:      aws.String(regionName),
	}))
	d.S3 = s3.New(sess)
	return d, nil
}

type readerCloser struct {
	io.Reader
}

func (r readerCloser) Close() error {
	return nil
}

// DownloadArtifact downloads the artifact at repo and sha to f.
func (d *S3Downloader) DownloadArtifact(f *os.File, repo, sha string) (int, error) {
	s3Path := fmt.Sprintf("%v/%v.tgz", repo, sha)
	input := &s3.GetObjectInput{}
	input.SetBucket(d.Bucket).SetKey(s3Path)
	var w io.WriterAt = f
	if d.MaxTransferRate > 0 {
		rateLimitWriter := newRateLimitWriterAt(d.MaxTransferRate, w)
		w = rateLimitWriter
		defer rateLimitWriter.Close()
	}
	downloader := s3manager.NewDownloaderWithClient(d.S3, func(downloader *s3manager.Downloader) {
		downloader.Concurrency = d.Concurrency
	})
	ctx, cancel := context.WithTimeout(context.Background(), d.Timeout)
	defer cancel()
	nr, err := downloader.DownloadWithContext(ctx, w, input)
	if err != nil {
		return int(nr), err
	}
	return int(nr), nil
}
