package artifact

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
)

type testS3SingleGet struct {
	t *testing.T
	s3iface.S3API
	Bucket string
	Key string
	Content string
}

type closingReader struct {
	io.Reader
}
func (c closingReader) Close() error {
	return nil
}

func (c *testS3SingleGet) GetObjectWithContext(ctx aws.Context, input *s3.GetObjectInput, opts ... request.Option) (*s3.GetObjectOutput, error) {
	if input.Bucket == nil {
		c.t.Fatal("input.Bucket was nil")
	}
	if input.Key == nil {
		c.t.Fatal("input.Key was nil")
	}
	if *input.Bucket == c.Bucket && *input.Key == c.Key {
		l := int64(len(c.Content))
		return &s3.GetObjectOutput{
			Body: closingReader{ bytes.NewReader([]byte(c.Content)) },
			ContentLength: &l,
	 }, nil
	}
	return nil, fmt.Errorf("object not found, bucket: %s key: %s", *input.Bucket, *input.Key)
}

// Checks that S3Downloader.Download writes the correct content
func TestS3DownloaderGetReaderContent(t *testing.T) {
	testBucket, testPath, testContent := "courier-test", "foo/bar/1234.tgz", "hello"
	testRepo, testSHA := "foo/bar", "1234"
	s := &testS3SingleGet{
		t: t,
		Bucket: testBucket,
		Key: testPath,
		Content: testContent,
	}
	downloader := &S3Downloader{
		Bucket: testBucket,
		S3:     s,
	}
	tarFile, err := ioutil.TempFile("", "courier")
	if err != nil {
		t.Fatal(err)
	}
	defer os.Remove(tarFile.Name())
	_, err = downloader.DownloadArtifact(tarFile, testRepo, testSHA)
	if err != nil {
		t.Fatalf("S3Downloader.Download had an error on getting repo %s sha %s: %v", testRepo, testSHA, err)
	}
	tarFile.Seek(0, 0)
	content, err := ioutil.ReadAll(tarFile)
	if err != nil {
		t.Fatal(err)
	}
	if string(content) != testContent {
		t.Errorf("Reader content is different from test content\nGot: %s\nExpected: %s", content, testContent)
	}
}
