package config

import (
	"fmt"
	"strconv"
)

const (
	// ErrUnsupportedType is returned when a type is requested that cannot be converted
	ErrUnsupportedType = "unsupported type"
)

type stringError string

func (s stringError) Error() string {
	return string(s)
}

// Config is a JSON object that has been marshalled into a map[string]interface{}, with wrapper functions for checking and retrieving values.
type Config map[string]interface{}

// IntValue returns the int value of key.
func (c Config) IntValue(key string) (ret int, err error) {
	return c.IntValueOrDefault(key, 0)
}

// IntValueOrDefault returns the string value of key. If the the config is unset at key, then def is returned.
func (c Config) IntValueOrDefault(key string, def int) (ret int, err error) {
	if !c.Has(key) {
		ret = def
		return
	}
	switch value := c[key].(type) {
	case float64:
		ret = int(value)
	case string:
		ret, err = strconv.Atoi(value)
	default:
		err = stringError(ErrUnsupportedType)
	}
	if err != nil {
		err = fmt.Errorf("could not read int value of %s: %v", key, err)
	}
	return
}

// StringValue returns the string value of key.
func (c Config) StringValue(key string) (ret string, err error) {
	return c.StringValueOrDefault(key, "")
}

// StringValueOrDefault returns the string value of key. If the the config is unset at key, then def is returned.
func (c Config) StringValueOrDefault(key, def string) (ret string, err error) {
	if !c.Has(key) {
		ret = def
		return
	}
	switch value := c[key].(type) {
	case string:
		ret = value
	default:
		err = stringError(ErrUnsupportedType)
	}
	if err != nil {
		err = fmt.Errorf("could not read string value of %s: %#v", key, err)
	}
	return
}

// Has returns whether the key is in the object.
func (c Config) Has(key string) bool {
	_, ok := c[key]
	return ok
}
