package config

import (
	"encoding/json"
	"testing"
)

type expectErrorTest func(*testing.T, error)

func expectNilError(t *testing.T, err error) {
	if err != nil {
		t.Errorf("Got error %v, expected nil", err)
	}
}

func createTestConfig(t *testing.T, j string) Config {
	var c = make(Config)
	if err := json.Unmarshal([]byte(j), &c); err != nil {
		t.Fatalf("Could not create config: %v", err)
	}
	return c
}

func testIntValue(t *testing.T, j, key string, expectedValue int, errTester expectErrorTest) {
	c := createTestConfig(t, j)
	ret, err := c.IntValue(key)
	if ret != expectedValue {
		t.Errorf("Returned value %d, expected value %d", ret, expectedValue)
	}
	errTester(t, err)
}

func testIntValueOrDefault(t *testing.T, j, key string, def, expectedValue int, errTester expectErrorTest) {
	c := createTestConfig(t, j)
	ret, err := c.IntValueOrDefault(key, def)
	if ret != expectedValue {
		t.Errorf("Returned value %d, expected value %d", ret, expectedValue)
	}
	errTester(t, err)
}

func testStringValue(t *testing.T, j, key, expectedValue string, errTester expectErrorTest) {
	c := createTestConfig(t, j)
	ret, err := c.StringValue(key)
	if ret != expectedValue {
		t.Errorf("Returned value %s, expected value %s", ret, expectedValue)
	}
	errTester(t, err)
}

func testStringValueOrDefault(t *testing.T, j, key, def, expectedValue string, errTester expectErrorTest) {
	c := createTestConfig(t, j)
	ret, err := c.StringValueOrDefault(key, def)
	if ret != expectedValue {
		t.Errorf("Returned value %s, expected value %s", ret, expectedValue)
	}
	errTester(t, err)
}

func TestIntValueParsedString(t *testing.T) {
	testIntValue(t, `{"a": "1"}`, "a", 1, expectNilError)
}

func TestIntValueUnparsed(t *testing.T) {
	testIntValue(t, `{"a": 1}`, "a", 1, expectNilError)
}

func TestIntValueUnset(t *testing.T) {
	testIntValue(t, `{}`, "a", 0, expectNilError)
}

func TestIntValueOrDefault(t *testing.T) {
	testIntValueOrDefault(t, `{"a": 1}`, "a", 1, 1, expectNilError)
}

func TestIntValueOrDefaultUnset(t *testing.T) {
	testIntValueOrDefault(t, `{}`, "a", 1, 1, expectNilError)
}

func TestStringValue(t *testing.T) {
	testStringValue(t, `{"a": "test1"}`, "a", "test1", expectNilError)
}

func TestStringValueUnset(t *testing.T) {
	testStringValue(t, `{}`, "a", "", expectNilError)
}

func TestStringValueOrDefault(t *testing.T) {
	testStringValueOrDefault(t, `{"a": "test"}`, "a", "default", "test", expectNilError)
}

func TestStringValueOrDefaultUnset(t *testing.T) {
	testStringValueOrDefault(t, `{}`, "a", "test", "test", expectNilError)
}
