package distribution

import (
	"fmt"
	"log"

	"code.justin.tv/release/courier/pkg/common"
	"code.justin.tv/release/courier/pkg/structs"
	"code.justin.tv/release/courier/pkg/tar"
)

func Execute(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner) (*structs.RemoteExecResult, error) {
	// TODO: have local install query target directory based on artifact to guarantee consistency
	targetName := ""
	var err error
	if _, ok := style.(*tar.Courier); ok {
		targetName, err = options.GetTargetName()
		if err != nil {
			return nil, err
		}
	}
	s := "fast"
	concurrency := uint(0)
	if options.DeployConfig != nil {
		if options.DeployConfig.Distribution != nil {
			s = *options.DeployConfig.Distribution.Style
			if options.DeployConfig.Distribution.Concurrency != nil {
				concurrency = *options.DeployConfig.Distribution.Concurrency
			}
		}
	}

	var result *structs.RemoteExecResult
	switch s {
	default:
		log.Printf("Unknown distribution style: %s. Falling back to fast", s)
		fallthrough
	case "fast":
		result, err = fast(options, style, targetName, commandRunner)
	case "gradual":
		result, err = gradual(concurrency, options, style, targetName, commandRunner)
	}
	if err != nil {
		return result, err
	}

	// Determine the distribution status
	if result.FailCnt > 0 {
		if options.ShortCircuit {
			err = fmt.Errorf("short-circuit tripped by errors on %v hosts", result.FailCnt)
		} else if result.FailCnt == len(options.Hosts) {
			err = fmt.Errorf("all hosts failed remote execution")
		} else if options.DeployConfig.Distribution != nil && options.DeployConfig.Distribution.FailThreshold != nil {
			t, v := options.GetThresholdValues("distribution")
			if t == "percentage" && v > 0 && common.CalculateFailRate(len(options.Hosts), result.FailCnt) >= v {
				err = fmt.Errorf("exceeded fail-threshold %v %%", v)
			} else if t == "number" && v > 0 && result.FailCnt >= int(v) {
				err = fmt.Errorf("exceeded fail-threshold %v", v)
			}
		}
	}

	return result, err
}
