package distribution

import (
	"fmt"
	"log"
	"sync"
	"time"

	"code.justin.tv/release/courier/pkg/structs"
)

func gradual(concurrency uint, options *structs.Options, style structs.Courier, targetName string, commandRunner structs.RemoteCommandRunner) (*structs.RemoteExecResult, error) {
	log.Printf("Beginning distribution of %v using 'gradual' style. concurrency: %d", options.Repo, concurrency)

	if concurrency < 1 {
		return nil, fmt.Errorf("invalid concurrency level %d", concurrency)
	}

	if len(options.Hosts) > 1 {
		halfFleet := uint(len(options.Hosts) / 2)
		if concurrency > halfFleet {
			log.Printf("Concurrency changed to %d since %d exceeds half of hosts", halfFleet, concurrency)
			concurrency = halfFleet
		}
	}

	cmd := style.RemoteInstallCmd(options)

	if targetName != "" {
		cmd = fmt.Sprintf("%s --%s %s", cmd, "target", targetName)
	}

	var wg sync.WaitGroup
	failHostsChan := make(chan *structs.FailHostInfo, len(options.Hosts))

	workQueue := make(chan string)
	for i := uint(0); i < concurrency; i++ {
		wg.Add(1)
		go func(options *structs.Options, queue chan string) {
			defer wg.Done()
			for host := range queue {
				err := commandRunner.RunRemoteHost(host, cmd, options, style.RemoteInstallFlags(options))
				if err != nil {
					failHostsChan <- &structs.FailHostInfo{host, err}
				}
			}
		}(options, workQueue)
	}

	for _, host := range options.Hosts {
		workQueue <- host

		// Without the sleep below, we seem to have issues where when we have more
		// than around 40 concurrent connections being started, some number of them
		// will time out and fail. This sleep presumably is preventing this from
		// happening by avoiding some sort of centralized resource contention.
		// - DS 2015-04-03
		time.Sleep(5 * time.Millisecond)
	}
	close(workQueue)

	wg.Wait()
	close(failHostsChan)

	result := &structs.RemoteExecResult{NumHosts: len(options.Hosts), FailCnt: 0, FailHosts: []*structs.FailHostInfo{}}
	for host := range failHostsChan {
		result.FailCnt++
		result.FailHosts = append(result.FailHosts, host)
	}

	return result, nil
}
