package executor

import (
	"fmt"
	"os"
	"os/exec"
	"path"
)

// NoExecutorFound is returned when SelectExecutor wasn't able to find
// any of the specified executors
type NoExecutorFound struct {
	executors []string
}

func (e *NoExecutorFound) Error() string {
	return fmt.Sprintf("Executor: no candidates found from: %s", e.executors)
}

// IsNoExecutorFound returns true if the error is the result of no
// executors being found by SelectExecutor
func IsNoExecutorFound(err error) bool {
	if _, ok := err.(*NoExecutorFound); ok {
		return true
	}
	return false
}

// SelectExecutor given a list of potential candidates returns the
// first valid candidate in a Executor struct
func SelectExecutor(dir string, executors ...string) (*Executor, error) {
	for _, exe := range executors {
		script := path.Join(dir, exe)
		fi, err := os.Stat(script)

		if os.IsNotExist(err) {
			continue
		}

		return &Executor{
			Script: script,
			info:   fi.Mode(),
			Dir:    dir,
		}, nil
	}
	return nil, &NoExecutorFound{executors}
}

// Executor contains information about the executable
type Executor struct {
	Script string
	info   os.FileMode
	Dir    string
}

// CourierCommand returns an exec.Cmd configured based on the
// executable state of the command, if it is not executable we shell
// out to bash
func (e *Executor) CourierCommand(out, err *os.File) (c *exec.Cmd) {
	if isExecutable(e.info) {
		c = exec.Command(e.Script)
	} else {
		c = exec.Command("bash", e.Script)
	}
	c.Stdout, c.Stderr = out, err
	c.Dir = e.Dir
	return
}

func isExecutable(mode os.FileMode) bool {
	return mode&0111 != 0
}
