package executor

import (
	"fmt"
	"io/ioutil"
	"os"
	"os/exec"
	"reflect"
	"testing"
)

func TestExecutorNoneFound(t *testing.T) {
	exe, err := SelectExecutor(os.TempDir(), "foo")
	if !IsNoExecutorFound(err) {
		t.Fatalf("Executor/NoneFound: expected failure, got: %v", exe)
	}
}

func testExecutor(name string, fm os.FileMode) (c *exec.Cmd, f *os.File, err error) {
	d, err := ioutil.TempDir("", "")
	if err != nil {
		return
	}

	f, err = os.OpenFile(
		fmt.Sprintf("%s/%s", d, name),
		os.O_RDWR|os.O_CREATE|os.O_TRUNC,
		fm,
	)
	if err != nil {
		return
	}
	defer os.RemoveAll(d)

	exe, err := SelectExecutor(d, name)
	if IsNoExecutorFound(err) {
		return
	}

	output, err := ioutil.TempFile("", "")
	if err != nil {
		return
	}
	c = exe.CourierCommand(output, output)
	return
}

func TestExecutorNotExecutable(t *testing.T) {
	c, f, err := testExecutor("restart.sh", 0666)
	if err != nil {
		t.Fatalf("Executor/Shell: unexpected error: %s", err)
	}

	if !reflect.DeepEqual(c.Args, []string{"bash", f.Name()}) {
		t.Fatalf("Executor/Shell: expected to call bash, got: %s", c.Args)
	}
}

func TestExecutorExecutable(t *testing.T) {
	c, f, err := testExecutor("restart", 0777)
	if err != nil {
		t.Fatalf("Executor/Shell: unexpected error: %s", err)
	}

	if !reflect.DeepEqual(c.Args, []string{f.Name()}) {
		t.Fatalf("Executor/Shell: expected to call %s directly, got: %s", f.Name(), c.Args)
	}
}
