package pack

import (
	"fmt"
	"strings"

	"code.justin.tv/release/courier/pkg/structs"
)

type argError struct {
	message string
}

func (a argError) Error() string {
	return a.message
}

// IsArgErr detect if this is an argument error
func IsArgErr(err error) bool {
	_, ok := err.(argError)
	return ok
}

// Courier keep track of this run
type Courier struct {
	User    string
	options *structs.Options
}

// NewCourier set up new courier struct
func NewCourier(options *structs.Options) (*Courier, error) {
	options.SkipRestart = true
	c := &Courier{
		User:    "jtv",
		options: options,
	}

	return c, nil
}

// RemoteInstallCmd Install command for remote hosts
func (c *Courier) RemoteInstallCmd(options *structs.Options) string {
	user := options.User
	if user == "" {
		user = c.User
	}
	return fmt.Sprintf("sudo setuidgid %s courier pkg install", user)
}

// RemoteInstallFlags Install flags for install command on remote hosts
func (c *Courier) RemoteInstallFlags(options *structs.Options) map[string]*string {
	m := make(map[string]*string)
	m["repo"] = &options.Repo
	m["environment"] = &options.Environment
	m["consul-host"] = &options.ConsulHost
	if options.User != "" {
		m["user"] = &options.User
	}
	if options.Sha != "" {
		m["sha"] = &options.Sha
	}
	return m
}

// LocalInstall installs the repos package from apt/yum at the requested sha.
// repo
// The repo that is being installed. e.g. "release/courier"
func (c *Courier) LocalInstall(repo, sha string) error {
	if repo == "" {
		return argError{"--repo can't be blank"}
	}
	if len(sha) < 7 {
		return argError{fmt.Sprintf("sha is too short: %s", sha)}
	}
	packageName, err := getPackageName(repo)
	if err != nil {
		return err
	}
	version := fmt.Sprintf("1.0.0~%s-0", sha[:7])
	packager, err := getPackager()
	if err != nil {
		return err
	}
	return packager.InstallPackage(packageName, version)
}

func getPackageName(repo string) (string, error) {
	if len(strings.Split(repo, "/")) != 2 {
		return "", argError{fmt.Sprintf("Repo is not in <organization>/<service> format: %s", repo)}
	}
	organization := strings.Split(repo, "/")[0]
	service := strings.Split(repo, "/")[1]
	return fmt.Sprintf("twitch-%s-%s", organization, service), nil
}

// LocalRemove removes the installed apt/yum package.
// repo
// The repo that is being removed. e.g. "release/courier"
func (c *Courier) LocalRemove(repo string) error {
	packageName, err := getPackageName(repo)
	if err != nil {
		return err
	}
	packager, err := getPackager()
	if err != nil {
		return err
	}
	return packager.RemovePackage(packageName)
}

// RemoteRestartCommand is present for interface compatibility but not implemented in pkg mode
// as the packages restart themselves after installation. SkipRestart is forced on in initialization
func (c *Courier) RemoteRestartCmd(*structs.Options) string {
	return ""
}

// RemoteRestartFlags is present for interface compatibility but not implemented in pkg mode
func (c *Courier) RemoteRestartFlags(options *structs.Options) map[string]*string {
	return map[string]*string{}
}
