package pack

import (
	"fmt"
	"log"
	"os/exec"
	"strings"
)

type Packager interface {
	InstallPackage(packageName, version string) error
	RemovePackage(packageName string) error
}

func getPackager() (Packager, error) {
	output, err := exec.Command("lsb_release", "-is").CombinedOutput()
	if err != nil {
		return nil, fmt.Errorf("Error detecting OS: %v\n%s", err, string(output))
	}
	id := strings.TrimSpace(string(output))
	switch id {
	case "Ubuntu":
		return &aptPackager{}, nil
	case "AmazonAMI":
		fallthrough
	case "CentOS":
		return &yumPackager{}, nil
	default:
		return nil, fmt.Errorf("Unrecognized OS ID: %s", id)
	}
}

type aptPackager struct{}

func (*aptPackager) InstallPackage(packageName, version string) error {
	log.Println("Updating package lists")
	cmd := exec.Command("apt-get", "-y", "update")
	output, err := cmd.CombinedOutput()
	log.Println(string(output))
	if err != nil {
		return fmt.Errorf("Could not update package manager: %v\n%s", err, string(output))
	}
	log.Println("Updated package lists")

	log.Println("Apt installing:", packageName, version)
	output, err = exec.Command("apt-get", "-y", "--allow-downgrades", "install", fmt.Sprintf("%s=%s", packageName, version)).CombinedOutput()
	if err != nil {
		return fmt.Errorf("apt-get install %s@%s failed: %v\n%s", packageName, version, err, string(output))
	}
	return nil
}

func (*aptPackager) RemovePackage(packageName string) error {
	log.Println("Apt removing:", packageName)
	output, err := exec.Command("apt-get", "-qy", "remove", packageName).CombinedOutput()
	if err != nil {
		return fmt.Errorf("apt-get remove %s failed: %v\n%s", packageName, err, string(output))
	}
	return nil
}

type yumPackager struct{}

func (*yumPackager) InstallPackage(packageName, version string) error {
	log.Println("Updating package lists")
	cmd := exec.Command("yum", "-y", "makecache")
	output, err := cmd.CombinedOutput()
	log.Println(string(output))
	if err != nil {
		return fmt.Errorf("Could not update package manager: %v\n%s", err, string(output))
	}
	log.Println("Updated package lists")

	log.Println("Yum installing:", packageName, version)

	// Yum complains if a version is "lower", and exits 0 without installing it. We have to both install
	// and downgrade to a specific version to guarantee that it actually installs
	output, err = exec.Command("yum", "-y", "install", fmt.Sprintf("%s-%s", packageName, version)).CombinedOutput()
	if err != nil {
		return fmt.Errorf("yum install %s@%s failed: %v\n%s", packageName, version, err, string(output))
	}
	output, err = exec.Command("yum", "-y", "downgrade", fmt.Sprintf("%s-%s", packageName, version)).CombinedOutput()
	if err != nil {
		return fmt.Errorf("yum downgrade %s@%s failed: %v\n%s", packageName, version, err, string(output))
	}
	return nil
}

func (*yumPackager) RemovePackage(packageName string) error {
	log.Println("yum removing:", packageName)
	output, err := exec.Command("yum", "-y", "remove", packageName).CombinedOutput()
	if err != nil {
		return fmt.Errorf("yum remove %s failed: %v\n%s", packageName, err, output)
	}
	return nil
}
