package restart

import (
	"log"
	"sync"
	"time"

	"code.justin.tv/release/courier/pkg/structs"
)

func fast(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner) (*structs.RemoteExecResult, error) {
	log.Printf("Beginning restart of %v using 'fast' style.", options.Repo)

	cmd := style.RemoteRestartCmd(options)

	var wg sync.WaitGroup
	failHostsChan := make(chan *structs.FailHostInfo, len(options.Hosts))

	for _, host := range options.Hosts {
		wg.Add(1)
		go func(options *structs.Options, host string) {
			defer wg.Done()
			err := commandRunner.RunRemoteHost(host, cmd, options, style.RemoteRestartFlags(options))
			if err != nil {
				failHostsChan <- &structs.FailHostInfo{host, err}
			}
		}(options, host)

		// Without the sleep below, we seem to have issues where when we have more
		// than around 40 concurrent connections being started, some number of them
		// will time out and fail. This sleep presumably is preventing this from
		// happening by avoiding some sort of centralized resource contention.
		// - DS 2015-04-03
		time.Sleep(5 * time.Millisecond)
	}

	wg.Wait()
	close(failHostsChan)

	result := &structs.RemoteExecResult{NumHosts: len(options.Hosts), FailCnt: 0, FailHosts: []*structs.FailHostInfo{}}
	for host := range failHostsChan {
		result.FailCnt++
		result.FailHosts = append(result.FailHosts, host)
	}

	return result, nil
}
