package restart

import (
	"fmt"
	"log"

	"code.justin.tv/release/courier/pkg/common"
	"code.justin.tv/release/courier/pkg/structs"
)

func Execute(options *structs.Options, style structs.Courier, commandRunner structs.RemoteCommandRunner) (*structs.RemoteExecResult, error) {
	s := "fast"
	concurrency := uint(0)
	if options.DeployConfig != nil {
		if options.DeployConfig.Restart != nil {
			s = *options.DeployConfig.Restart.Style
			if options.DeployConfig.Restart.Concurrency != nil {
				concurrency = *options.DeployConfig.Restart.Concurrency
			}
		}
	}

	var result *structs.RemoteExecResult
	var err error
	switch s {
	default:
		log.Printf("Unknown restart style: %s. Falling back to fast", s)
		fallthrough
	case "fast":
		result, err = fast(options, style, commandRunner)
	case "gradual":
		result, err = gradual(concurrency, options, style, commandRunner)
	}
	if err != nil {
		return result, err
	}

	// Determine the distribution status
	if result.FailCnt > 0 {
		if options.ShortCircuit {
			err = fmt.Errorf("short-circuit tripped by errors on %v hosts", result.FailCnt)
		} else if result.FailCnt == len(options.Hosts) {
			err = fmt.Errorf("all hosts failed remote execution")
		} else if options.DeployConfig.Restart != nil && options.DeployConfig.Restart.FailThreshold != nil {
			t, v := options.GetThresholdValues("restart")
			if t == "percentage" && v > 0 && common.CalculateFailRate(len(options.Hosts), result.FailCnt) >= v {
				err = fmt.Errorf("exceeded fail-threshold %v %%", v)
			} else if t == "number" && v > 0 && result.FailCnt >= int(v) {
				err = fmt.Errorf("exceeded fail-threshold %v", v)
			}
		}
	}

	return result, err
}
