package ssh

import (
	"fmt"
	"log"
	"net"
	"os"
	"sync"

	"golang.org/x/crypto/ssh/agent"
)

type Agent struct {
	agent.Agent
	once sync.Once
	err  error
	conn net.Conn
}

// Open will open a connection to the SSH agent in the global scope. This
// can then be used by all sub-processes. The ssh/agent package locks when
// talking to the connection.
func (ag *Agent) Open() error {
	ag.once.Do(func() {
		sshAuthSock := os.Getenv("SSH_AUTH_SOCK")

		if sshAuthSock == "" {
			log.Println("No SSH_AUTH_SOCK specified, skipping setup")
			return
		}

		ag.conn, ag.err = net.Dial("unix", sshAuthSock)
		if ag.err != nil {
			return
		}
		ag.Agent = agent.NewClient(ag.conn)
	})
	// We save the error and return it to all calls of OpenAgent, not just the
	// one that actually sets up the agent.
	if ag.err != nil {
		return fmt.Errorf("error connecting to ssh-agent: %v", ag.err)
	}

	return nil
}

func (ag *Agent) Close() error {
	if err := ag.conn.Close(); err != nil {
		return err
	}

	return nil
}

func (ag *Agent) IsSetup() bool {
	return (ag.Agent != nil)
}
