package ssh

import "testing"
import "time"
import (
	"code.justin.tv/release/courier/pkg/structs"
	"strings"
)

func TestSplitPort(t *testing.T) {
	h, p := splitPort("reltool2.sfo01.justin.tv")
	if h != "reltool2.sfo01.justin.tv" || p != 22 {
		t.Errorf("did not get expected host/port: %v:%v", h, p)
	}
}

func TestSplitPortCustom(t *testing.T) {
	h, p := splitPort("reltool2.sfo01.justin.tv:41958")
	if h != "reltool2.sfo01.justin.tv" || p != 41958 {
		t.Errorf("did not get expected host/port: %v:%v", h, p)
	}
}

func TestSplitPortInvalid(t *testing.T) {
	h, p := splitPort("reltool2.sfo01.justin.tv:foobar")
	if h != "reltool2.sfo01.justin.tv" || p != 22 {
		t.Errorf("did not get expected host/port: %v:%v", h, p)
	}
}

func TestMakeCommandString(t *testing.T) {
	f := make(map[string]*string)
	v1 := "val1"
	f["key1"] = &v1

	if MakeCommandString("testcmd", nil) != "testcmd" {
		t.Errorf("wrong return of MakeCommandString for nil")
	}

	if MakeCommandString("testcmd", f) != "testcmd --key1 \"val1\"" {
		t.Errorf("wrong return of MakeCommandString for %v", f)
	}
}

func TestConvertFlagsToParameterString(t *testing.T) {
	f := make(map[string]*string)
	v1 := "val1"
	v2 := "val2"

	if convertFlagsToParameterString(f) != "" {
		t.Errorf("wrong return of ConvertFlagsToParameterString for %v", f)
	}
	f["key1"] = nil
	if convertFlagsToParameterString(f) != " --key1" {
		t.Errorf("wrong return of ConvertFlagsToParameterString for %v", f)
	}
	f["key1"] = &v1
	if convertFlagsToParameterString(f) != " --key1 \"val1\"" {
		t.Errorf("wrong return of ConvertFlagsToParameterString for %v", f)
	}
	f["key2"] = nil
	s := convertFlagsToParameterString(f)
	if !(strings.Contains(s, " --key1 \"val1\"") && strings.Contains(s, " --key2")) {
		t.Errorf("wrong return of ConvertFlagsToParameterString for %v", f)
	}
	f["key2"] = &v2
	s = convertFlagsToParameterString(f)
	if !(strings.Contains(s, " --key1 \"val1\"") && strings.Contains(s, " --key2 \"val2\"")) {
		t.Errorf("wrong return of ConvertFlagsToParameterString for %v", f)
	}
}

type sshConfigTest struct {
	givenRetries    int
	expectedRetries int
	givenTimeout    int
	expectedTimeout int
	givenUser       string
	expectedUser    string
}

var sshConfigTests = []sshConfigTest{
	{13, 13, 55, 55, "john", "john"},
	{13, 13, 55, 55, "", "deploy"},                     // falls back to default `deploy` user
	{0, DefaultNumDialRetries, 55, 55, "john", "john"}, // fall back to default retries
	{13, 13, 0, DefaultSshDialTimeout, "john", "john"}, // falls back to default timeout
}

func TestSshClientConfig(t *testing.T) {
	for _, testCase := range sshConfigTests {
		expectedTimeoutDuration := time.Duration(testCase.expectedTimeout) * time.Second
		options := structs.Options{
			Retries:        testCase.givenRetries,
			SshDialTimeout: testCase.givenTimeout,
		}
		if err := sharedAgent.Open(); err != nil {
			t.Errorf("Failed to initialize ssh-agent - %v", err)
		}
		conf, err := sshClientConfig(testCase.givenUser, &options)
		if err != nil {
			t.Errorf("Expected no error, got %v", err)
		}
		if conf.User != testCase.expectedUser {
			t.Errorf("Expected %s as user, got %s", testCase.expectedUser, conf.User)
		}
		if conf.Timeout != expectedTimeoutDuration {
			t.Errorf("Expected %v as timeout, got %v", expectedTimeoutDuration, conf.Timeout)
		}
	}
}
