package structs

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/release/courier/pkg/common"
	"github.com/armon/consul-api"
	"github.com/cenkalti/backoff"
	"github.com/codegangsta/cli"
)

type RemoteExecResult struct {
	NumHosts  int
	FailCnt   int
	FailHosts []*FailHostInfo
}

type FailHostInfo struct {
	Hostname  string
	ErrString error
}

type Courier interface {
	RemoteInstallCmd(*Options) string
	RemoteInstallFlags(*Options) map[string]*string
	RemoteRestartCmd(*Options) string
	RemoteRestartFlags(*Options) map[string]*string
}

type RemoteCommandRunner interface {
	RunRemoteHost(host string, cmd string, options *Options, flags map[string]*string) error
}

type DeployedVersionUpdater interface {
	UpdateDeployedVersion(options *Options, goodVersion string) error
}

type Options struct {
	ConsulHost       string
	SkadiHost        string
	Hosts            []string
	Retries          int
	User             string
	Environment      string
	Repo             string
	ShortCircuit     bool
	SkipRestart      bool
	SkipSymlink      bool
	SymlinkInRestart bool

	// tar
	Dir    string
	Target string
	Sha    string

	// ssh
	SshDialTimeout int

	// per-deployer variables
	SkadiID uint64

	// deploy.json settings
	DeployConfig *api.DeployConfig

	// internal cache
	DeployedVersion  string
	KnownGoodVersion string

	// configuration
	ConsulFallbackDC string
}

func NewOptions(c *cli.Context) *Options {
	o := &Options{}

	// Set consul defaults
	o.ConsulHost = c.String("consul-host")
	o.ConsulFallbackDC = c.String("consul-fallback-dc")
	// Then override from config file if found
	err := o.loadConsulConfig(c.String("config-path"))
	if err != nil {
		log.Fatal(err)
	}
	// Then override if user specified
	if c.IsSet("consul-host") {
		o.ConsulHost = c.String("consul-host")
	}
	if c.IsSet("consul-fallback-dc") {
		o.ConsulFallbackDC = c.String("consul-fallback-dc")
	}

	o.SkadiHost = c.String("skadi-host")
	o.Hosts = common.SplitStringWithoutEmptyStr(c.String("hosts"), ",")
	if len(o.Hosts) == 0 {
		o.Hosts = common.SplitStringWithoutEmptyStr(readMultiEnvHosts(), ",")
	}

	o.Environment = c.String("environment")
	o.Repo = c.String("repo")
	o.ShortCircuit = c.Bool("short-circuit")
	o.SkipRestart = c.Bool("skip-restart")
	o.SkipSymlink = c.Bool("skip-symlink")
	o.SymlinkInRestart = c.Bool("symlink-in-restart")
	o.Retries = c.Int("retries")
	o.User = c.String("user")

	o.Dir = c.String("dir")
	o.Target = c.String("target")
	o.Sha = c.String("sha")
	o.KnownGoodVersion = c.String("rollback-sha")

	o.SshDialTimeout = c.Int("ssh-dial-timeout")

	// information to track deploy status, mainly used for skadi triggered deploy jobs
	o.SkadiID, _ = strconv.ParseUint(os.Getenv("SKADI_ID"), 10, 64)

	return o
}

func (o *Options) loadConsulConfig(configPath string) error {
	if configPath == "" {
		return nil
	}

	type Config struct {
		ConsulHost       string `json:"consul_host"`
		ConsulFallbackDC string `json:"consul_fallback_dc"`
	}
	log.Printf("ConfigPath: %v", configPath)
	data, err := ioutil.ReadFile(configPath)
	if err != nil {
		return err
	}
	var c Config
	err = json.Unmarshal(data, &c)
	if err != nil {
		return err
	}

	if c.ConsulHost != "" {
		o.ConsulHost = c.ConsulHost
	}
	if c.ConsulFallbackDC != "" {
		o.ConsulFallbackDC = c.ConsulFallbackDC
	}
	return nil
}

func (o *Options) LoadSkadiSettings() error {
	if o.DeployConfig == nil {
		skadiClient, err := api.NewClient(o.SkadiHost, nil)
		if err != nil {
			return err
		}

		parts := strings.SplitN(o.Repo, "/", 3)
		if len(parts) != 2 {
			return fmt.Errorf("--repo ($REPO) must be in format: '<owner>/<repo>'")
		}
		owner := parts[0]
		name := parts[1]

		ref, err := o.GetDeployedVersion()
		if err != nil {
			return err
		}
		// Retry skadi API operations, but gradually back off up to 3 minutes.
		retryNotify := func(err error, wait time.Duration) {
			log.Printf("Failed call to skadi while loading repository settings: (%v), retrying in %ds", err, wait/time.Second)
		}
		var settings *api.Settings
		retryOperation := func() error {
			settings, err = skadiClient.Repositories().Settings(owner, name, ref)
			return err
		}
		retryBackOff := backoff.NewExponentialBackOff()
		retryBackOff.MaxInterval = 15 * time.Second
		retryBackOff.MaxElapsedTime = 180 * time.Second
		err = backoff.RetryNotify(retryOperation, retryBackOff, retryNotify)
		if err != nil {
			return fmt.Errorf("error loading repository settings: %v", err)
		}
		o.DeployConfig = settings.Deploy
	}
	return nil
}

// GetTargetName will either return the target set on the command line
// or generate one based upon what the deployed version should be.
func (o *Options) GetTargetName() (string, error) {
	var sha string
	var err error
	if o.Target != "" {
		return o.Target, nil
	}
	sha = o.Sha
	if sha == "" {
		sha, err = o.GetDeployedVersion()
		if err != nil {
			return "", err
		}
	}
	return fmt.Sprintf("%v-%v", time.Now().Unix(), sha), nil
}

func (o *Options) GetDeployedVersion() (string, error) {
	// If we are handed a target by upstream. Check and see if it contains a
	// hash. If so we will attempt to deploy that version. If we can't parse it
	// then we continue with trying to load the value from consul.
	//
	// This is a hacky workaround to handle the differences between the
	// master/slave architecture and the ability for courier to both run
	// independently or as a slave.
	if o.Target != "" {
		v, err := common.ParseVersion(o.Target)
		if err == nil {
			return v, nil
		}
	}

	if o.DeployedVersion != "" {
		return o.DeployedVersion, nil
	}

	var err error
	o.DeployedVersion, err = o.getConsulKey("deployed-version")
	return o.DeployedVersion, err
}

func (o *Options) GetKnownGoodVersion() (string, error) {
	if o.KnownGoodVersion != "" {
		return o.KnownGoodVersion, nil
	}

	var err error
	o.KnownGoodVersion, err = o.getConsulKey("known-good-version")
	return o.KnownGoodVersion, err
}

func (o *Options) GetThresholdValues(section string) (string, float32) {
	const DEfAULT_THRESHOLD_TYPE = "number"
	const DEFAULT_THRESHOLD_VALUE = 0

	if o.DeployConfig == nil {
		return DEfAULT_THRESHOLD_TYPE, DEFAULT_THRESHOLD_VALUE
	}

	var failThresholdType *string
	var failThrehold *float32
	if section == "restart" {
		if o.DeployConfig.Restart == nil {
			return DEfAULT_THRESHOLD_TYPE, DEFAULT_THRESHOLD_VALUE
		}
		failThresholdType = o.DeployConfig.Restart.FailThresholdType
		failThrehold = o.DeployConfig.Restart.FailThreshold
	} else {
		if o.DeployConfig.Distribution == nil {
			return DEfAULT_THRESHOLD_TYPE, DEFAULT_THRESHOLD_VALUE
		}
		failThresholdType = o.DeployConfig.Distribution.FailThresholdType
		failThrehold = o.DeployConfig.Distribution.FailThreshold
	}

	t := "number"
	v := float32(0)
	if failThresholdType != nil && (*failThresholdType == "percentage" || *failThresholdType == "percent") {
		t = "percentage"
	}
	if failThrehold != nil && *failThrehold > 0 {
		v = *failThrehold
	}

	return t, v
}

func (o *Options) getConsulKey(prefix string) (string, error) {
	consulClient, err := consulapi.NewClient(&consulapi.Config{
		Address: o.ConsulHost,
	})
	if err != nil {
		return "", err
	}

	if o.Repo == "" {
		return "", fmt.Errorf("--repo can't be blank")
	}

	if o.Environment == "" {
		return "", fmt.Errorf("--environment can't be blank")
	}

	log.Printf("Querying %v of %v in %v", prefix, o.Repo, o.Environment)

	var kv *consulapi.KVPair
	// Retry consul API operations, but gradually back off up to 15 seconds
	retryNotify := func(err error, wait time.Duration) {
		log.Printf("Failed call to consul while getting kv: (%v), retrying in %ds", err, wait/time.Second)
	}
	retryOperation := func() error {
		var err error
		kv, _, err = consulClient.KV().Get(path.Join(prefix, o.Repo, o.Environment), nil)
		return err
	}
	retryBackOff := backoff.NewExponentialBackOff()
	retryBackOff.MaxInterval = 5 * time.Second
	retryBackOff.MaxElapsedTime = 60 * time.Second
	err = backoff.RetryNotify(retryOperation, retryBackOff, retryNotify)

	if kv != nil {
		sha := string(kv.Value)
		log.Printf("Version: %v", sha)

		return sha, nil
	}

	// Retry consul API operations against the fallback datacenter, but wait longer before giving up.
	retryNotify = func(err error, wait time.Duration) {
		log.Printf("Failed call to consul while getting kv from fallback datacenter: (%v), retrying in %ds", err, wait/time.Second)
	}
	retryOperation = func() error {
		var err error
		kv, _, err = consulClient.KV().Get(path.Join(prefix, o.Repo, o.Environment), &consulapi.QueryOptions{Datacenter: o.ConsulFallbackDC})
		return err
	}
	retryBackOff = backoff.NewExponentialBackOff()
	retryBackOff.MaxInterval = 20 * time.Second
	retryBackOff.MaxElapsedTime = 60 * time.Second
	err = backoff.RetryNotify(retryOperation, retryBackOff, retryNotify)
	if err != nil {
		return "", err
	}
	if kv != nil {
		sha := string(kv.Value)
		log.Printf("Version: %v", sha)

		return sha, nil
	}

	return "", fmt.Errorf("no version in consul for %v/%v", o.Repo, o.Environment)
}

func (o *Options) RemoveHosts(badHosts []string) error {
	badMap := make(map[string]bool)
	for _, host := range badHosts {
		badMap[host] = true
	}

	newHosts := []string{}
	for _, host := range o.Hosts {
		if badMap[host] {
			continue
		}
		newHosts = append(newHosts, host)
	}
	o.Hosts = newHosts
	return nil
}

func readMultiEnvHosts() string {
	hosts := os.Getenv("HOSTS")
	for i := 2; i < 10; i++ {
		moreHosts := os.Getenv(fmt.Sprintf("HOSTS%d", i))
		if moreHosts == "" {
			break
		}

		hosts += "," + moreHosts
	}
	return hosts
}

type ConsulVersionUpdater struct{}

func (cvu ConsulVersionUpdater) UpdateDeployedVersion(options *Options, goodVersion string) (err error) {
	consulClient, err := consulapi.NewClient(&consulapi.Config{
		Address: options.ConsulHost,
	})
	if err != nil {
		return
	}

	dcs, err := consulClient.Catalog().Datacenters()
	if err != nil {
		return
	}

	key := fmt.Sprintf("deployed-version/%v/%v", options.Repo, options.Environment)

	for _, dc := range dcs {
		_, err = consulClient.KV().Put(
			&consulapi.KVPair{Key: key, Value: []byte(goodVersion)},
			&consulapi.WriteOptions{Datacenter: dc},
		)
		if err != nil {
			return
		}
	}
	return
}

type CommandResult string

const (
	COMMAND_SUCCESS CommandResult = "SUCCESS"
	COMMAND_FAILURE CommandResult = "FAILURE"
	COMMAND_NOOP    CommandResult = "NOOP"
)

type CourierEvent struct {
	ConsulHost           string `json:"consul_host"`
	SkadiHost            string `json:"skadi_host"`
	NumRemoteHosts       int    `json:"num_remote_hosts"`
	NumFailedRemoteHosts int    `json:"num_failed_remote_hosts"`
	Retries              int    `json:"retries"`
	User                 string `json:"user"`
	Environment          string `json:"environment"`
	Repo                 string `json:"repository"`
	ShortCircuit         bool   `json:"short_circuit"`
	SkipRestart          bool   `json:"skip_restart"`
	SkipSymlink          bool   `json:"skip_symlink"`
	SymlinkInRestart     bool   `json:"symlink_in_restart"`
	Dir                  string `json:"directory"`
	Target               string `json:"target"`
	Sha                  string `json:"sha"`
	SshDialTimeout       int    `json:"ssh_dial_timeout"`
	SkadiID              uint64 `json:"skadi_id"`
	DeployedVersion      string `json:"deployed_version"`
	KnownGoodVersion     string `json:"known_good_version"`
	Command              string `json:"command"`
	CourierVersion       string `json:"courier_version"`
	CommandStartTime     int64  `json:"command_start_time"`
	CommandEndTime       int64  `json:"command_end_time"`
	LocalHostName        string `json:"local_host_name"`
	LocalRun             bool   `json:"local_run"`
	CommandResult        string `json:"command_result"`
}

func NewCourierEvent(command string, hostname string, localRun bool, numRemoteFailedHosts int, commandStartTime time.Time, commandEndTime time.Time, commandResult CommandResult, version string, options *Options) *CourierEvent {

	return &CourierEvent{
		ConsulHost:           options.ConsulHost,
		SkadiHost:            options.SkadiHost,
		NumRemoteHosts:       len(options.Hosts),
		NumFailedRemoteHosts: numRemoteFailedHosts,
		Retries:              options.Retries,
		User:                 options.User,
		Environment:          options.Environment,
		Repo:                 options.Repo,
		ShortCircuit:         options.ShortCircuit,
		SkipRestart:          options.SkipRestart,
		SkipSymlink:          options.SkipSymlink,
		SymlinkInRestart:     options.SymlinkInRestart,
		Dir:                  options.Dir,
		Target:               options.Target,
		Sha:                  options.Sha,
		SshDialTimeout:       options.SshDialTimeout,
		SkadiID:              options.SkadiID,
		DeployedVersion:      options.DeployedVersion,
		KnownGoodVersion:     options.KnownGoodVersion,
		Command:              command,
		CourierVersion:       version,
		CommandStartTime:     commandStartTime.Unix(),
		CommandEndTime:       commandEndTime.Unix(),
		LocalHostName:        hostname,
		LocalRun:             localRun,
		CommandResult:        string(commandResult),
	}
}
