package tar

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"regexp"
	"sort"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/release/courier/pkg/artifact"
	"code.justin.tv/release/courier/pkg/common"
	"code.justin.tv/release/courier/pkg/config"
	"code.justin.tv/release/courier/pkg/executor"
	"code.justin.tv/release/courier/pkg/structs"

	humanize "github.com/dustin/go-humanize"
	"github.com/nightlyone/lockfile"
)

const (
	currentDir            = "current"
	defaultBuildRetention = 5

	LockFilename      string        = "courier.lock"
	LockRetryInterval time.Duration = time.Duration(10 * time.Second)
	LockRetryTimeout  time.Duration = time.Duration(15 * time.Minute)
)

type argError struct {
	message string
}

func (a argError) Error() string {
	return a.message
}

// IsArgErr detect if this is an argument error
func IsArgErr(err error) bool {
	_, ok := err.(argError)
	return ok
}

// Courier keep track of this run
type Courier struct {
	KeepReleases int
	User         string
	options      *structs.Options
	artifact.Downloader
}

// NewCourier set up new courier struct
func NewCourier(options *structs.Options) (*Courier, error) {
	c := &Courier{
		User:    "jtv",
		options: options,
	}

	return c, nil
}

// LoadConfig load the courier configuration file
func (c *Courier) LoadConfig(configPath string) error {
	log.Printf("ConfigPath: %v", configPath)
	configFile, err := ioutil.ReadFile(configPath)
	if err != nil {
		return err
	}
	var configJSON config.Config
	if err = json.Unmarshal(configFile, &configJSON); err != nil {
		return err
	}
	if c.KeepReleases, err = configJSON.IntValueOrDefault("build_retention", defaultBuildRetention); err != nil {
		return err
	}
	if c.User, err = configJSON.StringValue("user"); err != nil {
		return err
	}

	var downloader string
	if downloader, err = configJSON.StringValueOrDefault("downloader", "s3"); err != nil {
		return err
	}
	if downloader == "s3" {
		if c.Downloader, err = artifact.LoadS3Downloader(configJSON); err != nil {
			return err
		}
	}
	if downloader == "http" {
		if c.Downloader, err = artifact.LoadHTTPDownloader(configJSON); err != nil {
			return err
		}
	}
	return nil
}

// RemoteInstallCmd Install command for remote hosts
func (c *Courier) RemoteInstallCmd(options *structs.Options) string {
	user := options.User
	if user == "" {
		user = c.User
	}
	return fmt.Sprintf("sudo setuidgid %s courier tar install", user)
}

// RemoteInstallFlags Install flags for install command on remote hosts
func (c *Courier) RemoteInstallFlags(options *structs.Options) map[string]*string {
	m := make(map[string]*string)
	m["repo"] = &options.Repo
	m["environment"] = &options.Environment
	m["dir"] = &options.Dir
	m["consul-host"] = &options.ConsulHost
	if options.SkipSymlink {
		m["skip-symlink"] = nil
	}
	if options.SymlinkInRestart {
		m["symlink-in-restart"] = nil
	}
	if options.User != "" {
		m["user"] = &options.User
	}
	if options.Sha != "" {
		m["sha"] = &options.Sha
	}
	return m
}

// RemoteRestartCmd Restart command for remote hosts
func (c *Courier) RemoteRestartCmd(options *structs.Options) string {
	user := options.User
	if user == "" {
		user = c.User
	}
	return fmt.Sprintf("sudo setuidgid %s courier tar restart", user)
}

// RemoteRestartFlags Restart flags for the restart command on remote hosts
func (c *Courier) RemoteRestartFlags(options *structs.Options) map[string]*string {
	m := make(map[string]*string)
	m["repo"] = &options.Repo
	m["environment"] = &options.Environment
	m["dir"] = &options.Dir
	if options.SkipSymlink {
		m["skip-symlink"] = nil
	}
	if options.SymlinkInRestart {
		m["symlink-in-restart"] = nil
	}
	if options.User != "" {
		m["user"] = &options.User
	}
	return m
}

// isSymlink takes a string and checks if the string is an absolute path.
// If string is absolute path, checks if file is a symlink
func isSymlink(file string) bool {
	f, err := os.Lstat(file)
	if err != nil {
		return false
	}

	// Bitwise operator used to determine symlinky-ness
	if f.Mode()&os.ModeSymlink == os.ModeSymlink {
		return true
	}

	return false
}

func getCurrentVersion(file string) (string, error) {
	if !filepath.IsAbs(file) {
		return "", fmt.Errorf("%q: is not an absolute path, unexpected results may occur", file)
	}

	// file not existing is a valid scenario in cases where the symlink was manually removed or fresh install
	if _, err := os.Stat(file); err != nil {
		// short circuiting here and sending err = nil since file not existing is a valid starting state
		return "", nil
	}

	// file existing and not a symlink
	if !isSymlink(file) {
		// sending error back since "current" should only be a symlink, other file types may need eyes on it
		return "", fmt.Errorf("%q exists but is not a symlink", file)
	}

	target, err := os.Readlink(file)
	if err != nil {
		return "", err
	}

	return common.ParseVersion(target)
}

// LocalInstall downloads the tar from S3 and extracts it to the local machine,
// updating a "current" symlink to point to the extracted directory.
//
// repo
// The repo that is being installed. e.g. "release/courier"
//
// environment
// The environment that is being deployed. e.g. "clean-staging"
//
// baseDir
// The root dir to install the artifact into. e.g. "/opt/twitch"
//
// target
// The "name" of the directory we are going to deploy into. This is generated
// upstream on the master to ensure that all nodes get the same "name".
// Since this name is a combination of a timestamp and the sha allowing each
// host to generate it would allow different names on different boxes.
func (c *Courier) LocalInstall(repo, environment, baseDir, sha, target, configpath string, skipsymlink bool, symlinkinrestart bool) (bool, error) {
	err := c.CheckArgs(repo, baseDir, target, sha, environment)
	noop := true
	if err != nil {
		return noop, err
	}

	if configpath == "" {
		configpath = "/etc/courier.conf"
	}

	err = c.LoadConfig(configpath)
	if err != nil {
		return noop, err
	}

	// Make sure no other courier runs on the same directory during the run
	if err := os.MkdirAll(baseDir, 0755); err != nil {
		return noop, err
	}
	if lock, err := tryPidLock(filepath.Join(baseDir, LockFilename), LockRetryTimeout); err != nil {
		return noop, err
	} else {
		defer lock.Unlock()
	}

	currentversion, err := getCurrentVersion(filepath.Join(baseDir, currentDir))
	if err != nil {
		return noop, err
	}

	if currentversion == sha {
		log.Println("Current symlink is same as current version from consul...doing nothing")
		return noop, nil
	}

	// operation is no longer noop, there is work to do
	noop = false

	// See if sha already exists in releases dir
	// If it does, symlink to it and run the cleanup

	// releasesDir is the directory that we install releases into
	releasesDir := filepath.Join(baseDir, "releases")
	extractDir := filepath.Join(releasesDir, target)
	pattern := fmt.Sprintf("*-%s", sha)

	releases := []os.FileInfo{}

	if _, err := os.Stat(releasesDir); err == nil {
		releases, err = ioutil.ReadDir(releasesDir)
		if err != nil {
			return noop, err
		}
	}

	for _, release := range releases {
		matched, err := filepath.Match(pattern, release.Name())
		if err != nil {
			return noop, err
		}
		if matched {
			targetDir := filepath.Join(releasesDir, release.Name())
			// release already exists, skipping download
			if !symlinkinrestart && !skipsymlink {
				err = c.LocalSymlink(baseDir, targetDir)
				if err != nil {
					return noop, err
				}
			}
			return noop, nil
		}
	}

	tarFile, err := ioutil.TempFile("", "courier")
	if err != nil {
		return noop, err
	}
	defer os.Remove(tarFile.Name())

	err = c.downloadArtifact(tarFile, repo, sha)
	if err != nil {
		return noop, err
	}

	err = os.MkdirAll(filepath.Join(baseDir, "shared"), 0755)
	if err != nil {
		return noop, err
	}

	err = c.extractRelease(extractDir, tarFile)
	if err != nil {
		return noop, err
	}

	err = c.moveSignedManifest(baseDir, extractDir, sha)
	if err != nil {
		return noop, err
	}

	err = c.runPostInstall(environment, extractDir)
	if err != nil {
		return noop, err
	}

	if !symlinkinrestart && !skipsymlink {
		err = c.LocalSymlink(baseDir, extractDir)
		if err != nil {
			return noop, err
		}
	}

	return noop, nil
}

// LocalSymlink set up the symlink to the current release
func (c *Courier) LocalSymlink(baseDir, target string) error {

	err := c.symlinkRelease(baseDir, target)
	if err != nil {
		return err
	}

	err = c.cleanupOldReleases(baseDir)
	if err != nil {
		return err
	}

	err = checkSymlink(path.Join(baseDir, currentDir), target)
	if err != nil {
		return err
	}
	return nil
}

// CheckArgs check arguments to make sure they aren't blank
func (c *Courier) CheckArgs(repo, baseDir, target, sha, environment string) error {
	if repo == "" {
		return argError{"--repo can't be blank"}
	}
	if baseDir == "" {
		return argError{"--dir can't be blank"}
	}
	if target == "" {
		return argError{"--target can't be blank"}
	}
	if sha == "" {
		return argError{"sha can't be blank"}
	}
	if environment == "" {
		return argError{"environment can't be blank"}
	}
	return nil
}

// LocalRestart restart the release pointed by the current symlink
func (c *Courier) LocalRestart(Options *structs.Options, repo, environment, baseDir, sha, target string, symlinkInRestart bool) error {

	err := c.CheckArgs(repo, baseDir, target, sha, environment)
	if err != nil {
		return err
	}

	// This lock is just to guarantee the package to remain intact during the restart process.
	if lock, err := tryPidLock(filepath.Join(baseDir, LockFilename), LockRetryTimeout); err != nil {
		return err
	} else {
		defer lock.Unlock()
	}

	if symlinkInRestart {
		releases := []os.FileInfo{}
		releasesDir := filepath.Join(baseDir, "releases")
		if _, err := os.Stat(releasesDir); err == nil {
			releases, err = ioutil.ReadDir(releasesDir)
			if err != nil {
				return err
			}
		}

		pattern := fmt.Sprintf("*-%s", sha)
		found := false
		for _, release := range releases {
			matched, err := filepath.Match(pattern, release.Name())
			if err != nil {
				return err
			}
			if matched {
				found = true
				targetDir := filepath.Join(releasesDir, release.Name())
				// release already exists, skipping download
				err = c.LocalSymlink(baseDir, targetDir)
				if err != nil {
					return err
				}
				break
			}
		}
		if !found {
			return fmt.Errorf("LocalRestart: symlinkInRestart is set, but no previously installed releases directory is found!")
		}
	}
	p := path.Join(baseDir, currentDir, "courier")
	restartNames := []string{"restart.sh", "restart"}
	if Options.DeployConfig.Restart != nil && Options.DeployConfig.Restart.Service != nil {
		for _, restartName := range restartNames {
			restartPath := filepath.Join(p, restartName)
			_, err := os.Stat(restartPath)
			if err != nil {
				continue
			}

			log.Printf("found restart script %q, however service is being restarted directly due to \"service\" config value", restartPath)
		}

		err := c.restartService(Options.DeployConfig.Restart)
		if err != nil {
			return err
		}
	} else {

		restarter, err := executor.SelectExecutor(p, restartNames...)
		if err != nil {
			return err
		}

		// We need to execute the restart script from the baseDir and not
		// baseDir/current/courier.
		restarter.Dir = baseDir

		log.Printf("Executing %v", restarter.Script)

		cmd := restarter.CourierCommand(os.Stdout, os.Stderr)
		err = cmd.Run()
		if err != nil {
			return err
		}
	}

	return nil
}

func (c *Courier) LocalStatus(repo, environment, baseDir, sha, target string) (string, error) {
	err := c.CheckArgs(repo, baseDir, target, sha, environment)
	if err != nil {
		return "", err
	}

	// Make sure no other courier runs on the same directory during the run
	if err := os.MkdirAll(baseDir, 0755); err != nil {
		return "", err
	}
	if lock, err := tryPidLock(filepath.Join(baseDir, LockFilename), LockRetryTimeout); err != nil {
		return "", err
	} else {
		defer lock.Unlock()
	}

	currentVersion, err := getCurrentVersion(filepath.Join(baseDir, currentDir))
	if err != nil {
		return "", err
	}

	if currentVersion != sha {
		return "", fmt.Errorf("current installed version '%s' doesn't match version from consul '%s'", currentVersion, sha)
	}

	return currentVersion, nil
}

// restartService finds matching daemontools-managed services, restarts them,
// and returns an error if the service does not remain up for long enough.
func (c *Courier) restartService(restart *api.DeployRestartConfig) error {
	wait := 10
	if restart.Wait != nil {
		wait = int(*restart.Wait)
	}

	configUptime := wait / 2
	if restart.Uptime != nil {
		restartUptime := int(*restart.Uptime)
		if restartUptime > wait {
			log.Printf("Using default Uptime (%d) since provided Uptime is greater than Wait", configUptime)
		} else {
			configUptime = restartUptime
		}
	}

	flag := "-i"
	if restart.Signal != nil {
		flag = *restart.Signal
	}

	services, err := filepath.Glob(fmt.Sprintf("/etc/service/%v", *restart.Service))
	if err != nil {
		return err
	}

	for _, service := range services {
		cmd := exec.Command("sudo", "svc", flag, service)
		err := cmd.Run()
		if err != nil {
			return err
		}

		time.Sleep(time.Duration(wait) * time.Second)

		var out bytes.Buffer
		cmd = exec.Command("sudo", "svstat", service)
		cmd.Stdout = &out
		err = cmd.Run()
		if err != nil {
			return err
		}

		r := regexp.MustCompile(".* up \\(pid [0-9]*\\) ([0-9]*) seconds.*")
		matches := r.FindStringSubmatch(out.String())
		if len(matches) != 2 {
			return fmt.Errorf("unabled to parse service uptime from %v", out.String())
		}

		serviceUptime, err := strconv.Atoi(matches[1])
		if err != nil {
			return err
		}

		if serviceUptime < configUptime {
			return fmt.Errorf("service %v failed uptime check using %d wait: got %d, expected %d", service, wait, serviceUptime, configUptime)
		}
	}

	return nil
}

func (c *Courier) downloadArtifact(tarFile *os.File, repo, sha string) error {
	log.Printf("Downloading %v/%v to %v", repo, sha, tarFile.Name())
	startTime := time.Now()
	bw, err := c.DownloadArtifact(tarFile, repo, sha)
	duration := time.Since(startTime)
	totalSize, totalUnit := humanize.ComputeSI(float64(bw))
	rateSize, rateUnit := humanize.ComputeSI(float64(bw) / duration.Seconds())
	var durationFormatted string
	if s := duration.Seconds(); int64(s) > 60 {
		m := int(duration.Minutes())
		durationFormatted = fmt.Sprintf("%dm%ds ", m, int(s)-m*60)
	} else {
		durationFormatted = fmt.Sprintf("%3.1fs", s)
	}
	if err != nil {
		return fmt.Errorf("Download error: %v; %4.1f %sB in %s (%4.1f %sB/s)", err, totalSize, totalUnit, durationFormatted, rateSize, rateUnit)
	}
	log.Printf("Download complete, %4.1f %sB in %s (%4.1f %sB/s)", totalSize, totalUnit, durationFormatted, rateSize, rateUnit)
	return nil
}

func (c *Courier) extractRelease(dir string, tarFile *os.File) error {
	tempDir, err := ioutil.TempDir("", "releasesTemp")
	if err != nil {
		return err
	}
	// ensure we cleanup the tempDir, ignoring any errors because the below os.Rename should be removing it
	defer os.RemoveAll(tempDir)

	cmd := exec.Command("tar", "-pxzf", tarFile.Name())
	cmd.Stdout, cmd.Stderr = os.Stdout, os.Stderr
	cmd.Dir = tempDir
	err = cmd.Run()
	if err != nil {
		return err
	}

	// tar extraction updates the timestamp of extractDir, so we touch it to enforce it as "newest"
	err = os.Chtimes(tempDir, time.Now(), time.Now())
	if err != nil {
		return err
	}

	// move successfully extracted dir to final location
	err = os.RemoveAll(dir)
	if err != nil {
		return err
	}

	err = os.MkdirAll(path.Dir(dir), 0755)
	if err != nil {
		return err
	}

	err = os.Rename(tempDir, dir)
	if err != nil {
		return err
	}

	return nil
}

// if *.manifest and *.manifest.hash are present, move them to baseDir/manifests
func (c *Courier) moveSignedManifest(baseDir, extractDir, sha string) error {
	manifestDir := filepath.Join(baseDir, "manifests")
	filename := sha + ".manifest"
	signed := filename + ".hash"
	err := os.MkdirAll(manifestDir, 0755)
	if err != nil {
		return err
	}

	_, err = os.Stat(filepath.Join(extractDir, filename))
	if err != nil {
		return nil
	}

	_, err = os.Stat(filepath.Join(extractDir, signed))
	if err != nil {
		return nil
	}

	err = os.Rename(filepath.Join(extractDir, filename), filepath.Join(manifestDir, filename))
	if err != nil {
		return err
	}

	err = os.Rename(filepath.Join(extractDir, signed), filepath.Join(manifestDir, signed))
	if err != nil {
		return err
	}

	return nil
}

func (c *Courier) runPostInstall(environment, dir string) error {
	p := path.Join(dir, "courier")
	postInstaller, err := executor.SelectExecutor(p, "post-install.sh")
	if executor.IsNoExecutorFound(err) {
		return nil
	}
	// We need to execute the runPostInstall script from the extract dir and not
	// extractDir/courier.
	postInstaller.Dir = dir

	log.Printf("Executing %v", postInstaller.Script)

	cmd := postInstaller.CourierCommand(os.Stdout, os.Stderr)
	cmd.Env = []string{"CURRENT_RELEASE=" + dir, "ENVIRONMENT=" + environment}
	err = cmd.Run()
	if err != nil {
		return err
	}

	return nil
}

func (c *Courier) symlinkRelease(baseDir, extractDir string) error {
	linkDirTest := filepath.Join(baseDir, "deploytest")
	linkDir := filepath.Join(baseDir, currentDir)

	// Error if extractDir does not exist to prevent dead symlinks
	_, err := os.Lstat(extractDir)
	if err != nil {
		return err
	}

	// Test creation of symlink, if it succeeds, rename to "current"
	err = os.Symlink(extractDir, linkDirTest)
	if err != nil {
		return err
	}

	err = os.Rename(linkDirTest, linkDir)
	if err != nil {
		return err
	}
	log.Printf("Symlinking %q to %q", linkDir, extractDir)

	return nil
}

// checkSymlink checks the symlink linkSrc and makes sure that it
// matches with testPath.
func checkSymlink(linkSrc, testPath string) error {
	// Error if linkSrc does not exist

	linkInfo, err := os.Lstat(linkSrc)
	if err != nil {
		return fmt.Errorf("Could not get fileinfo of %s: %v", linkSrc, err)
	}
	if linkInfo.Mode()&os.ModeSymlink == 0 {
		return fmt.Errorf("%s is not a symlink", linkSrc)
	}
	linkDest, err := os.Readlink(linkSrc)
	if err != nil {
		return fmt.Errorf("Could not read link %s: %v", linkSrc, err)
	}
	destInfo, err := os.Stat(linkDest)
	if err != nil {
		return fmt.Errorf("Could not get fileinfo of %s (symlink from %s): %v", linkDest, linkSrc, err)
	}
	testInfo, err := os.Stat(testPath)
	if err != nil {
		return fmt.Errorf("Could not get fileinfo of %s: %v", testPath, err)
	}
	if !os.SameFile(destInfo, testInfo) {
		return fmt.Errorf("%s (symlink to %s) does not match release %s", linkSrc, linkDest, testPath)
	}
	return nil
}

type byModTimeDesc []os.FileInfo

func (t byModTimeDesc) Len() int {
	return len(t)
}

func (t byModTimeDesc) Swap(i, j int) {
	t[i], t[j] = t[j], t[i]
}

func (t byModTimeDesc) Less(i, j int) bool {
	return t[i].ModTime().After(t[j].ModTime())
}

func contains(str string, slc []string) bool {
	for _, s := range slc {
		if s == str {
			return true
		}
	}
	return false
}

func (c *Courier) cleanupOldReleases(dir string) error {
	knownGoodVersion, err := c.options.GetKnownGoodVersion()
	if err != nil {
		// Note that if we can't find a good known version, we just continue,
		// we don't return the error here. This is to avoid a failed
		// deployment in case there is something funky with the last good
		// version info.
		log.Printf("Couldn't find latest known good version to keep.")
	}
	currentDirLink, err := os.Readlink(filepath.Join(dir, currentDir))
	if err == nil {
		log.Printf("Found current symlink pointing to: %v - not deleting", currentDirLink)
	} else {
		currentDirLink = ""
	}
	rm := ReleasesMaintenance{c.KeepReleases, filepath.Join(dir, "releases"), filepath.Join(dir, "manifests"), knownGoodVersion, currentDirLink}

	err = rm.cleanup()
	if err == nil {
		err = rm.removeDanglingManifests()
	}
	return err
}

type ReleasesMaintenance struct {
	KeepReleases     int
	ReleasesDir      string
	ManifestsDir     string
	KnownGoodVersion string
	CurrentDirLink   string
}

func (rm *ReleasesMaintenance) cleanup() error {
	runningReleases, err := rm.findRunningReleases()
	if err != nil {
		return err
	}
	releases, err := ioutil.ReadDir(rm.ReleasesDir)
	if err != nil {
		return err
	}
	sort.Sort(byModTimeDesc(releases))
	for i, release := range releases {
		releaseName := release.Name()
		releasePath := filepath.Join(rm.ReleasesDir, releaseName)
		if contains(releaseName, runningReleases) {
			log.Printf("Currently running release: %s", releasePath)
		} else if rm.KnownGoodVersion != "" && strings.Contains(releaseName, rm.KnownGoodVersion) {
			log.Printf("Keeping last good version: %s", releasePath)
		} else if i < rm.KeepReleases {
			log.Printf("Keeping release: %s", releasePath)
		} else if rm.CurrentDirLink != "" && releasePath == rm.CurrentDirLink {
			log.Printf("Keeping symlinked release: %s", releasePath)
		} else {
			rm.removeReleaseFiles(releasePath, releaseName)
		}
	}
	return nil
}
func (rm *ReleasesMaintenance) removeReleaseFiles(releasePath string, releaseName string) {
	msg := fmt.Sprintf("Removing old release: %s", releasePath)
	os.RemoveAll(releasePath)
	releaseTokens := strings.Split(releaseName, "-")
	if len(releaseTokens) != 2 {
		log.Printf("Invalid release format: %s (%s)", releaseName, releasePath)
	} else {
		releaseHash := releaseTokens[1]
		found, err := filepath.Glob(filepath.Join(rm.ManifestsDir, "./"+releaseHash+".manifest*"))
		if err == nil && len(found) > 0 {
			msg += fmt.Sprintf(" - Also removing %d matching manifest files: %s", len(found), strings.Join(found, ", "))
			for _, manifestPath := range found {
				os.Remove(manifestPath)
			}

		}
	}
	log.Print(msg)
}

func (rm *ReleasesMaintenance) removeDanglingManifests() error {
	var releaseHash string
	manifests, err := ioutil.ReadDir(rm.ManifestsDir)
	if err != nil {
		return err
	}
	/* Remove all manifests without a matching build */
	for _, manifestFile := range manifests {
		manifestName := manifestFile.Name()
		if strings.HasSuffix(manifestName, ".manifest") {
			releaseHash = strings.TrimSuffix(manifestName, ".manifest")
		} else if strings.HasSuffix(manifestName, ".manifest.hash") {
			releaseHash = strings.TrimSuffix(manifestName, ".manifest.hash")
		} else {
			log.Printf("Unknown file type in manifests directory: %v", manifestName)
			continue
		}
		found, _ := filepath.Glob(filepath.Join(rm.ReleasesDir, "./*-"+releaseHash))
		if len(found) == 0 {
			manifestPath := filepath.Join(rm.ManifestsDir, manifestName)
			log.Printf("Removing manifest of unknown build: %v", manifestPath)
			os.Remove(manifestPath)
		}
	}

	return nil
}

func (rm *ReleasesMaintenance) findRunningReleases() ([]string, error) {
	cmd := exec.Command("sudo", "lsof", "-dcwd", "-Fn")
	stdout, stderr := new(bytes.Buffer), new(bytes.Buffer)
	cmd.Stdout, cmd.Stderr = stdout, stderr

	err := cmd.Run()
	if err != nil {
		io.Copy(os.Stderr, stderr)
		return nil, err
	}

	evalReleaseDir, err := filepath.EvalSymlinks(rm.ReleasesDir)
	if err != nil {
		return nil, err
	}
	absReleaseDir, err := filepath.Abs(evalReleaseDir)
	if err != nil {
		return nil, err
	}

	var runningReleases []string
	for {
		line, err := stdout.ReadString(byte('\n'))
		if err != nil && err != io.EOF {
			return nil, err
		}
		if len(line) == 0 {
			break
		}
		switch line[0] {
		case 'n':
			cwd := line[1:]
			if strings.HasPrefix(cwd, absReleaseDir) {
				relCwd, err := filepath.Rel(absReleaseDir, cwd)
				if err != nil {
					continue
				}
				dir := strings.TrimSpace(filepath.SplitList(relCwd)[0])
				if !contains(dir, runningReleases) {
					log.Printf("found %s.", strings.TrimSpace(dir))
					runningReleases = append(runningReleases, dir)
				}
			}
		}
		if err == io.EOF {
			break
		}
	}
	return runningReleases, nil
}

func tryPidLock(filepath string, timeout time.Duration) (*lockfile.Lockfile, error) {
	l, err := lockfile.New(filepath)
	if err != nil {
		return nil, err
	}

	numAttempt := 0
	for startTime := time.Now(); numAttempt == 0 || time.Since(startTime) < timeout; numAttempt++ {
		err := l.TryLock()
		if err == nil {
			return &l, nil
		}
		if !(err == lockfile.ErrBusy || err == lockfile.ErrNotExist) {
			return nil, err
		}
		log.Printf("Failed to acquire an exclusive lock on %v (will retry until timeout)- %v", filepath, err)

		if timeout == 0 {
			break
		}
		time.Sleep(LockRetryInterval)
	}

	// This is a safey logic to let it exit from unknown edge cases. Should never meet.
	return nil, fmt.Errorf("Maximum timeout reached but still couldn't acquire lock on %v", filepath)
}
