package tar

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"syscall"
	"testing"

	"code.justin.tv/release/courier/pkg/structs"
)

// TestLocalInstall is for testing that LocalInstall fails if the correct flags
// aren't passed in. This feels like I'm writing Ruby.
func TestLocalInstallErrs(t *testing.T) {
	type localInstallTest struct {
		testName         string
		repo             string
		environment      string
		dir              string
		sha              string
		target           string
		configpath       string
		skipsymlink      bool
		symlinkinrestart bool
	}

	dir := os.TempDir()

	tests := []localInstallTest{
		{
			testName:         "emptyRepo",
			repo:             "",
			environment:      "bar",
			dir:              dir,
			sha:              "abcd",
			target:           "blahblah",
			configpath:       "",
			skipsymlink:      false,
			symlinkinrestart: false,
		},
		{
			testName:         "emptyEnvironment",
			repo:             "foo",
			environment:      "",
			dir:              dir,
			sha:              "abcd",
			target:           "blahblah",
			configpath:       "",
			skipsymlink:      false,
			symlinkinrestart: false,
		},
		{
			testName:         "emptyDir",
			repo:             "foo",
			environment:      "bar",
			dir:              "",
			sha:              "abcd",
			target:           "blahblah",
			configpath:       "",
			skipsymlink:      false,
			symlinkinrestart: false,
		},
		{
			testName:         "emptySha",
			repo:             "foo",
			environment:      "bar",
			dir:              dir,
			sha:              "",
			target:           "blahblah",
			configpath:       "",
			skipsymlink:      false,
			symlinkinrestart: false,
		},
		{
			testName:         "emptyTarget",
			repo:             "foo",
			environment:      "bar",
			dir:              dir,
			sha:              "abcd",
			target:           "",
			configpath:       "",
			skipsymlink:      false,
			symlinkinrestart: false,
		},
	}
	options := structs.Options{}
	c, err := NewCourier(&options)
	if err != nil {
		t.Fatalf("Error creating NewCourier: %v", err)
	}
	for _, test := range tests {
		_, err := c.LocalInstall(test.repo, test.environment, test.dir, test.sha, test.target, test.configpath, test.skipsymlink, test.symlinkinrestart)
		if err == nil {
			t.Errorf("%q: Want err; got nil", test.testName)
		}
		if !IsArgErr(err) {
			t.Errorf("%q: Want invalid argument error; got %v", test.testName, err)
		}
	}
}

type TestDownloader func(*os.File, string, string) (int, error)

func (d TestDownloader) DownloadArtifact(f *os.File, repo, sha string) (int, error) {
	return d(f, repo, sha)
}

type ClosingBuffer struct {
	*bytes.Buffer
}

func (cb *ClosingBuffer) Close() error {
	return nil
}

func TestLoadConfig(t *testing.T) {
	c := &Courier{}
	var testConfig = `
	{"access_key":"AAAAAAAAAAAAAAAAAAAA","build_retention":"5","http_proxy":"proxy.somewhere","s3_bucket":"somes3bucket","secret_key":"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"}
	`
	testFile, err := ioutil.TempFile("", "couriercfgtest")
	if err != nil {
		t.Fatal(err)
	}
	_, err = testFile.WriteString(testConfig)
	if err != nil {
		t.Fatal(err)
	}
	defer os.Remove(testFile.Name())
	c.LoadConfig("couriercfgtest")
}

// Checks that the file contents written by downloadArtifact matches the reader output
func TestDownloadArtifactFileContents(t *testing.T) {
	var testContent = `this is a test.
this is only a test.`
	testFile, err := ioutil.TempFile("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.Remove(testFile.Name())
	c := &Courier{
		Downloader: TestDownloader(func(f *os.File, repo, sha string) (int, error) {
			return f.Write([]byte(testContent))
		}),
	}
	c.downloadArtifact(testFile, "", "")
	b, err := ioutil.ReadFile(testFile.Name())
	if err != nil {
		t.Fatal(err)
	}
	if string(b) != testContent {
		t.Errorf("downloadArtifact file content doesn't match artifact downloader reader content")
	}
}

// Checks that the parameters passed to GetReader by downloadArtifact are correct
func TestDownloadArtifactFileGetReaderParameters(t *testing.T) {
	var testRepo, testSha string = "org/repo", "1234567890abcdef"
	testFile, err := ioutil.TempFile("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.Remove(testFile.Name())
	c := &Courier{
		Downloader: TestDownloader(func(f *os.File, repo, sha string) (int, error) {
			if repo != testRepo || sha != testSha {
				t.Errorf("downloadArtifact GetReader parameters repo, sha(%s, %s) don't match expected (%s, %s)", repo, sha, testRepo, testSha)
			}
			return 0, nil
		}),
	}
	c.downloadArtifact(testFile, testRepo, testSha)
}

// Checks that an error is returned by checkSymlink when linkSrc doesn't exist
func TestCheckSymlinkSrcDoesntExist(t *testing.T) {
	var srcDir, destDir string = "current", "release"
	testDir, err := ioutil.TempDir("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.RemoveAll(testDir)
	if err := os.Mkdir(path.Join(testDir, destDir), 0755); err != nil {
		t.Fatal(err)
	}

	if err := checkSymlink(path.Join(testDir, srcDir), path.Join(testDir, destDir)); err == nil {
		t.Errorf("checkSymlink returned no error for source link that doesn't exist")
	}
}

// Checks that an error is returned by checkSymlink when testPath doesn't exist
func TestCheckSymlinkTestDoesntExist(t *testing.T) {
	var srcDir, destDir string = "current", "release"
	testDir, err := ioutil.TempDir("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.RemoveAll(testDir)
	if err := os.Symlink(path.Join(testDir, destDir), path.Join(testDir, srcDir)); err != nil {
		t.Fatal(err)
	}

	if err := checkSymlink(path.Join(testDir, srcDir), path.Join(testDir, destDir)); err == nil {
		t.Error("checkSymlink returned no error for testPath that doesn't exist")
	}
}

// Checks that no error is returned by checkSymlink when current points to release
func TestCheckSymlinkSrcPointsToTest(t *testing.T) {
	var srcDir, destDir string = "current", "release"
	testDir, err := ioutil.TempDir("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.RemoveAll(testDir)
	if err := os.Mkdir(path.Join(testDir, destDir), 0755); err != nil {
		t.Fatal(err)
	}
	if err := os.Symlink(path.Join(testDir, destDir), path.Join(testDir, srcDir)); err != nil {
		t.Fatal(err)
	}

	if err := checkSymlink(path.Join(testDir, srcDir), path.Join(testDir, destDir)); err != nil {
		t.Errorf("checkSymlink returned an error when linkSrc points to testPath: %v", err)
	}
}

// Checks that an error is returned by checkSymlink when linkSrc doesn't point to testPath
func TestCheckSymlinkSrcDoesntPointToTest(t *testing.T) {
	var srcDir, destDir, badDestDir string = "current", "release", "not-release"
	testDir, err := ioutil.TempDir("", "couriertest")
	if err != nil {
		t.Fatal(err)
	}
	defer os.RemoveAll(testDir)
	if err := os.Mkdir(path.Join(testDir, destDir), 0755); err != nil {
		t.Fatal(err)
	}
	if err := os.Symlink(path.Join(testDir, badDestDir), path.Join(testDir, srcDir)); err != nil {
		t.Fatal(err)
	}

	if err := checkSymlink(path.Join(testDir, srcDir), path.Join(testDir, destDir)); err == nil {
		t.Error("checkSymlink returned no error when current points to release directory")
	}
}

func TestTryPidLock(t *testing.T) {
	path := filepath.Join("/tmp", LockFilename)
	os.Remove(path) // force remove possible residue

	for i := 0; i < 3; i++ {
		l, err := tryPidLock(path, 0)
		if err != nil {
			t.Fatal(err)
		}

		err = l.Unlock()
		if err != nil {
			t.Fatal(err)
		}
	}
}

func TestTryPidLock_LockedByAnother(t *testing.T) {
	path := filepath.Join("/tmp", LockFilename)
	os.Remove(path) // force remove possible residue

	// Find another active pid
	pid2 := os.Getpid() - 1
	for ; pid2 > 0; pid2-- {
		proc, err := os.FindProcess(pid2)
		if err != nil {
			continue
		}
		if err := proc.Signal(syscall.Signal(0)); err == nil {
			break
		}
	}
	if pid2 == 0 {
		t.Fatal(fmt.Errorf("Failed to find another pid???"))
	}

	// lock file created by pid2
	f, err := os.Create(path)
	if err != nil {
		t.Fatal(err)
	}
	f.WriteString(fmt.Sprintf("%d\n", pid2))
	f.Close()

	// tryPidLock should fail
	l, err := tryPidLock(path, 0)
	if err == nil {
		t.Fatal(fmt.Sprintf("tryPidLock succeeded where expected to fail"))
	}

	// pid2 removed lock file
	os.Remove(path)

	// tryPidLock should succeed
	l, err = tryPidLock(path, 0)
	if err != nil {
		t.Fatal(err)
	}

	err = l.Unlock()
	if err != nil {
		t.Fatal(err)
	}
}

func TestTryPidLock_LockedByDead(t *testing.T) {
	path := filepath.Join("/tmp", LockFilename)
	os.Remove(path) // force remove possible residue

	// Find an inactive pid
	pid2 := os.Getpid() - 1
	for ; pid2 > 0; pid2-- {
		proc, err := os.FindProcess(pid2)
		if err != nil {
			continue
		}
		if err := proc.Signal(syscall.Signal(0)); err != nil {
			break
		}
	}
	if pid2 == 0 {
		t.Fatal(fmt.Errorf("Failed to find inactive pid???"))
	}

	// lock file created by pid2
	f, err := os.Create(path)
	if err != nil {
		t.Fatal(err)
	}
	f.WriteString(fmt.Sprintf("%d\n", pid2))
	f.Close()

	// tryPidLock should succeed
	l, err := tryPidLock(path, 0)
	if err != nil {
		t.Fatal(err)
	}

	err = l.Unlock()
	if err != nil {
		t.Fatal(err)
	}
}
