// +build integration

// The above blank line is required!!!
//
// Integration Tests will only run if you specify the -tags=integration flag to
// go test.
package tests

import (
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"os"
	"os/exec"
	"path/filepath"
	"strings"
	"testing"
	"time"

	"code.justin.tv/release/courier/tests/setup"
)

const (
	// TODO(tarrant 2013-11-19): This should be a test project of some form
	// that release controls.
	repo       = "release/courier-integration-tar"
	env        = "integration"
	configPath = "courier.conf"
)

var bin string

func TestMain(m *testing.M) {
	setup.Setup()
	bin = setup.Bin
	m.Run()
	setup.Teardown()
}

func sanityCheckTargetDir(t *testing.T, baseDir, targetDir string) {
	currentDir := filepath.Join(baseDir, "current")

	// Test that dir/releases/<target> exists
	targetDirInfo, err := os.Stat(targetDir)
	if err != nil {
		t.Fatalf("Error opening %q: %v", targetDir, err)
	}
	if !targetDirInfo.Mode().IsDir() {
		t.Fatalf("Target dir %q is not a directory.", targetDir)
	}

	// and is a subdirectory of dir/releases:
	parentDir := filepath.Dir(targetDir)
	if parentDir != filepath.Join(baseDir, "releases") {
		t.Errorf("%q is not a subdir of %q", targetDir, parentDir)
	}

	// Test if symlink points to dir/releases/<target>
	res, err := os.Readlink(currentDir)
	if err != nil {
		t.Fatalf(`Unable to access %q directory doesn't exist: %v`, currentDir, err)
	}

	if res != targetDir {
		t.Fatalf("symlink for current -> %q; want %q", res, targetDir)
	}
}

func TestShaIntegrationWithTarget(t *testing.T) {
	target := "foobar66"
	sha := "8c517a19abaf65bc7a4ee498bbd559173241d5af"

	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--target", target, "--sha", sha); err != nil {
		t.Fatal(err)
	}

	targetDir := filepath.Join(dir, "releases", target)

	sanityCheckTargetDir(t, dir, targetDir)
}

func TestShaIntegrationAndSymlinkInRestart(t *testing.T) {
	sha := "8c517a19abaf65bc7a4ee498bbd559173241d5af"

	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}

	courierdirpath := filepath.Join(dir, "courier")

	if _, err := os.Stat(courierdirpath); os.IsNotExist(err) {
		os.Mkdir(courierdirpath, 0755)
	}

	restartpath := filepath.Join(courierdirpath, "restart.sh")
	restartcontent := []byte("#!/usr/bin/bash\nexit 0\n")
	err = ioutil.WriteFile(restartpath, restartcontent, 0755)
	if err != nil {
		t.Fatal(err)
	}

	if err := runIntegration("tar", "install", bin, repo, env, dir, "--sha", sha, "--skip-symlink"); err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "restart", bin, repo, env, dir, "--sha", sha, "--symlink-in-restart"); err != nil {
		t.Fatal(err)
	}

	currentSymlink := filepath.Join(dir, "current")
	_, err = os.Stat(currentSymlink)
	if err != nil {
		t.Fatalf("should have created a symlink at %q", currentSymlink)
	}
}

func TestShaIntegrationWithoutTarget(t *testing.T) {
	sha := "8c517a19abaf65bc7a4ee498bbd559173241d5af"
	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--sha", sha); err != nil {
		t.Fatal(err)
	}

	matches, err := filepath.Glob(filepath.Join(dir, "releases", fmt.Sprintf("*-%s", sha)))
	if err != nil {
		t.Fatal(err)
	}
	if len(matches) != 1 {
		t.Fatalf("Had to find 1 matching directory, instead found: %+v", matches)
	}
	targetDir := matches[0]
	sanityCheckTargetDir(t, dir, targetDir)
}

func TestTargetIntegration(t *testing.T) {
	target := "foobar66"

	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--target", target); err != nil {
		t.Fatal(err)
	}

	targetDir := filepath.Join(dir, "releases", target)

	sanityCheckTargetDir(t, dir, targetDir)
}

func TestHTTPIntegration(t *testing.T) {
	t.Skip("DTA-4390 Re-enable when dp-dev is back.")
	httpConfigPath := "courier-http.conf"
	target := "foobar66"

	dir, err := ioutil.TempDir("", "http")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--target", target, "--config-path", httpConfigPath); err != nil {
		t.Fatal(err)
	}

	targetDir := filepath.Join(dir, "releases", target)

	sanityCheckTargetDir(t, dir, targetDir)
}

func TestNoTargetIntegration(t *testing.T) {
	dir, err := ioutil.TempDir("", "no-target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir); err != nil {
		t.Fatal(err)
	}

	releasesDir := filepath.Join(dir, "releases")

	files, err := ioutil.ReadDir(releasesDir)
	if err != nil {
		t.Fatal(err)
	}

	// This accomplishes several things. It lets us validate that only the
	// directory that we want got created. In addition it helps us ensure that
	// the /releases/ dir wasn't overwritten.
	if len(files) != 1 {
		t.Fatalf("Invalid number of items in releases dir %q: got %d; want %d", releasesDir, len(files), 1)
	}

	t.Logf("Target name: %q", files[0].Name())
	targetDir := filepath.Join(dir, "releases", files[0].Name())
	sanityCheckTargetDir(t, dir, targetDir)
}

func TestMultipleIntegration(t *testing.T) {
	dir, err := ioutil.TempDir("", "multiple")
	if err != nil {
		t.Fatal(err)
	}

	count := 6
	for i := 0; i <= count; i++ {
		if err := runIntegration("tar", "install", bin, repo, env, dir); err != nil {
			t.Fatal(err)
		}
		// TODO: We should be specifically be cloning the last <x> number of releases rather than sleeping.
		time.Sleep(1 * time.Second)
	}

	releasesDir := filepath.Join(dir, "releases")

	files, err := ioutil.ReadDir(releasesDir)
	if err != nil {
		t.Fatal(err)
	}

	// Courier will not download the same artifact multiple times.
	// Courier is called multiple times but should only download once giving a list
	// of one unique sha.
	shas := make(map[string]struct{})
	for _, finfo := range files {
		fname := strings.SplitN(finfo.Name(), "-", 2)
		if len(fname) != 2 {
			t.Fatalf("Invalid contents in relases dir: %q", releasesDir)
		}

		shas[fname[1]] = struct{}{}
	}

	// This accomplishes several things. It lets us validate that only the
	// directory that we want got created.
	if len(files) != len(shas) {
		t.Fatalf(
			"Invalid number of items in releases dir %q: got %d; want %d",
			releasesDir,
			len(files),
			len(shas),
		)
	}

	// Pick the current target:
	targetIndex := 0
	for i, _ := range files {
		if files[i].ModTime().UnixNano() > files[targetIndex].ModTime().UnixNano() {
			targetIndex = i
		}
	}

	t.Logf("Target name: %q", files[targetIndex].Name())
	targetDir := filepath.Join(dir, "releases", files[targetIndex].Name())
	sanityCheckTargetDir(t, dir, targetDir)
}

func TestNoSymlinkIntegration(t *testing.T) {
	target := "nolink"

	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--target", target, "--skip-symlink"); err != nil {
		t.Fatal(err)
	}

	currentSymlink := filepath.Join(dir, "current")

	_, err = os.Stat(currentSymlink)
	if err == nil {
		t.Fatalf("should not have created a symlink at %q", currentSymlink)
	}
}

func TestWithSymlinkIntegration(t *testing.T) {
	target := "withlink"

	dir, err := ioutil.TempDir("", "target")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir, "--target", target); err != nil {
		t.Fatal(err)
	}

	currentSymlink := filepath.Join(dir, "current")

	_, err = os.Stat(currentSymlink)
	if err != nil {
		t.Fatalf("should have created a symlink at %q", currentSymlink)
	}
}

func TestManifest(t *testing.T) {
	dir, err := ioutil.TempDir("", "manifest")
	if err != nil {
		t.Fatal(err)
	}
	if err := runIntegration("tar", "install", bin, repo, env, dir); err != nil {
		t.Fatal(err)
	}

	files, err := filepath.Glob(filepath.Join(dir, "manifests/*.manifest*"))
	if err != nil {
		t.Fatal(err)
	}

	log.Printf("%+v", files)

	if len(files) != 2 {
		t.Fatalf("expected 2 manifest files, found %v", len(files))
	}
}

type waitReader struct {
	waitChan chan struct{}
}

func (r *waitReader) Read(p []byte) (n int, err error) {
	<-r.waitChan
	return 0, io.EOF
}

func runTarInstallHelper(t *testing.T, runningDir string, dir string, additionalFlags ...string) {
	cmd := exec.Command("cat")
	cmd.Dir = runningDir
	cmdIn := new(waitReader)
	cmdIn.waitChan = make(chan struct{})
	cmd.Stdin = cmdIn
	cmd.Start()
	if err := runIntegration("tar", "install", bin, repo, env, dir, additionalFlags...); err != nil {
		t.Fatal(err)
	}
	cmdIn.waitChan <- struct{}{}
	_, err := os.Stat(runningDir)
	if err != nil {
		t.Fatalf("Unable to open running directory %q: %v", runningDir, err)
	}
}

func loadManifestsFixtures(t *testing.T, dir string, dontRemove []string) (runningDir string, removablePaths []string) {
	buildsToTest := 14
	buildRetention := 5
	/* default value, update when needed */
	path := filepath.Join(dir, "manifests")
	err := os.MkdirAll(path, 0755)
	if err != nil {
		t.Fatalf("Couldn't create directory %s: %s", path, err)
	}
	mtime := time.Date(1980, time.August, 24, 15, 0, 0, 0, time.UTC)
	/* create fake dirs simulating a bunch of builds with their manifests */
	for i := 0; i < buildsToTest; i++ {
		numStr := fmt.Sprintf("%02d", i)
		path = filepath.Join(dir, "releases", "x-"+numStr)
		if err = os.MkdirAll(path, 0755); err != nil {
			t.Fatalf("Couldn't create directory %s: %s", path, err)
		}
		if err = os.Chtimes(path, mtime, mtime.Add(time.Duration(i)*time.Hour)); err != nil {
			t.Fatalf("Couldn't change mtime of %s: %s", path, err)
		}

		if i == buildsToTest-1 {
			/* consider the most recent build the running build */
			runningDir = path
		} else if i < buildsToTest-buildRetention {
			canRemove := true
			for _, s := range dontRemove {
				if numStr == s {
					canRemove = false
				}
			}
			if canRemove {
				/* old build directories should be gone */
				removablePaths = append(removablePaths, path)

				/* old manifests should be gone too */
				path = filepath.Join(dir, "manifests", numStr+".manifest")
				_, err = os.Create(path)
				if err != nil {
					t.Fatalf("Couldn't create test manifest %s: %s", path, err)
				}
				removablePaths = append(removablePaths, path)
				path = filepath.Join(dir, "manifests", numStr+".manifest.hash")
				_, err = os.Create(path)
				if err != nil {
					t.Fatalf("Couldn't create test manifest %s: %s", path, err)
				}
				removablePaths = append(removablePaths, path)
			}
		}
	}
	/* add a couple of manifest files without a matching build directory */
	path = filepath.Join(dir, "manifests", "Kappa.manifest")
	_, err = os.Create(path)
	if err != nil {
		t.Fatalf("Couldn't create test manifest %s: %s", path, err)
	}
	removablePaths = append(removablePaths, path)
	path = filepath.Join(dir, "manifests", "PogChamp.manifest.hash")
	_, err = os.Create(path)
	if err != nil {
		t.Fatalf("Couldn't create test manifest %s: %s", path, err)
	}
	removablePaths = append(removablePaths, path)
	return
}

func TestRunningReleases(t *testing.T) {
	var path string
	dir, err := ioutil.TempDir("", "RunningRelease-")
	if err != nil {
		t.Fatalf("Couldn't create temporary directory: %s", err)
	}
	runningDir, removablePaths := loadManifestsFixtures(t, dir, []string{})
	runTarInstallHelper(t, runningDir, dir)

	/* assert that we actually removed the old/expired directories */
	for i, _ := range removablePaths {
		path = removablePaths[i]
		_, err = os.Stat(path)
		if err == nil {
			t.Fatalf("File/directory wasn't removed correctly: %v", path)
		}
	}

}

func TestKeepRollbackRelease(t *testing.T) {
	var path string
	dir, err := ioutil.TempDir("", "KeepRollback-")
	if err != nil {
		t.Fatalf("Couldn't create temporary directory: %s", err)
	}
	runningDir, removablePaths := loadManifestsFixtures(t, dir, []string{"03"})
	runTarInstallHelper(t, runningDir, dir, "--rollback-sha", "03")

	/* assert that we actually removed the old/expired directories */
	for i, _ := range removablePaths {
		path = removablePaths[i]
		_, err = os.Stat(path)
		if err == nil {
			t.Fatalf("File/directory wasn't removed correctly: %v", path)
		}
	}
}

func runIntegration(style, sub, bin, repo, env, dir string, additionalFlags ...string) error {
	parts := strings.Split(sub, " ")
	args := []string{style}
	args = append(args, parts[0])
	if len(parts) > 1 {
		args = append(args, parts[1])
	}
	args = append(args, []string{
		"--repo", repo,
		"--environment", env,
		"--dir", dir,
		"--consul-host", "api.us-west-2.prod.consul.live-video.a2z.com",
		"--config-path", configPath,
	}...)
	args = append(args, additionalFlags...)
	cmd := exec.Command(bin, args...)
	cmd.Stdout, cmd.Stderr = os.Stdout, os.Stderr
	if err := cmd.Run(); err != nil {
		return fmt.Errorf("Error running courier: %v", err)
	}
	return nil
}
