package solidsqs_test

import (
	"code.justin.tv/common/solidsqs"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"os"
	"sort"
	"time"
)

const (
	TestQueueURL    = "https://sqs.us-west-2.amazonaws.com/043714768218/solidsqs_test"
	TestQueueRegion = "us-west-2"
)

func Example() {
	// Init
	s, err := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)}, nil)
	if err != nil {
		fmt.Println(err)
		return
	}

	// Check connection
	length := s.Len()
	if length < 0 {
		fmt.Println("Queue is not reachable.", TestQueueURL)
		return
	} else if length > 0 {
		// Purge queue for testing
		if err := s.Purge(); err != nil {
			fmt.Println(err)
			return
		}
	}

	// Push
	if num, err := s.Push("insert1", "insert2", "insert3"); err != nil {
		fmt.Println(err)
		return
	} else {
		fmt.Println("pushed:", num)
	}

	// Pop
	// SQS could return less than it has but SolidSQS tries even queue have
	// more than requested, To prevent, unnecessary user-land loop, Pop()
	// tries to retrieve all data until `maxnum` reached or queue gets empty.
	// In this case, `pullsec` is only used at the very first internal call.
	if values, err := s.Pop(3, 1); err == nil {
		// values inserted at almost same time could be in mixed order
		sort.Strings(values)
		fmt.Println("popped:", len(values), values)
		return
	}

	// Output:
	// pushed: 3
	// popped: 3 [insert1 insert2 insert3]
}

func Example_simple() {
	s, _ := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)}, nil)
	if s.Len() < 0 {
		fmt.Println("Queue unreachable.")
		return
	}
	num, _ := s.Push("insertA", "insertB", "insertC")
	values, _ := s.Pop(num, 1)
	sort.Strings(values)
	fmt.Println(num, len(values), values)

	// Output:
	// 3 3 [insertA insertB insertC]
}

func ExampleSolidSQS_New() {
	// Log file for logging low-level queue activities
	out, err := os.OpenFile("/tmp/solidsqs_example.out", os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644)
	if err != nil {
		fmt.Println(err)
		return
	}

	// New with full custom configuration.
	s, _ := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)},
		&solidsqs.Config{
			RetryIntervalMin:   (100 * time.Millisecond),
			RetryIntervalMax:   (500 * time.Millisecond),
			RetryRuntime:       (1000 * time.Millisecond),
			AsynchronousDelete: false,
			DebugOut:           out,
		})

	num1, _ := s.Push("insert11", "insert12", "insert13")
	num2, _ := s.Push("insert14", "insert15")
	values, _ := s.Pop(5, 1)
	sort.Strings(values)

	fmt.Println(num1, num2, values)
	// Output:
	// 3 2 [insert11 insert12 insert13 insert14 insert15]
}

func ExampleSolidSQS_Pop() {
	s, _ := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)},
		&solidsqs.Config{
			RetryIntervalMin:   (100 * time.Millisecond),
			RetryIntervalMax:   (500 * time.Millisecond),
			RetryRuntime:       (1000 * time.Millisecond),
			AsynchronousDelete: true, // Delete messages in background
		})

	num, _ := s.Push("insert21", "insert22", "insert23", "insert24", "insert25")
	values, _ := s.Pop(5, 1)
	sort.Strings(values)
	fmt.Println(num, len(values), values)

	// Wait for the async deletion completed.
	for s.GetAsyncDeleteLen() > 0 {
		time.Sleep(100 * time.Millisecond)
	}

	// Output:
	// 5 5 [insert21 insert22 insert23 insert24 insert25]
}

func ExampleSolidSQS_PushWithDelay() {
	s, _ := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)}, nil)

	// Push with 1 second of delay
	num, _ := s.PushWithDelay(1, "insert31", "insert32", "insert33", "insert34", "insert35")

	// Should return nothing
	time.Sleep(100 * time.Millisecond)
	values1, _ := s.Pop(5, 0)
	fmt.Println(num, len(values1), values1)

	// Should return all
	time.Sleep(1 * time.Second)
	values2, _ := s.Pop(5, 0)
	sort.Strings(values2)
	fmt.Println(num, len(values2), values2)

	// Output:
	// 5 0 []
	// 5 5 [insert31 insert32 insert33 insert34 insert35]
}

func ExampleSolidSQS_Purge() {
	s, _ := solidsqs.New(TestQueueURL, &aws.Config{Region: aws.String(TestQueueRegion)}, nil)
	// Might be a good idea to check the length of the queue before calling
	// Purge() since purge operation is expensive on SQS side and limited to
	// be called 1 per 60 seconds
	if s.Len() > 0 {
		if err := s.Purge(); err != nil {
			fmt.Println(err)
		}
	}
}
