// Package solidsqs provides simplified methods to use of AWS SQS.
// It's intended to provide a set of helper methods that covers the most of
// common use cases in reliable way.
//
// SolidSQS treats and generalizes SQS as a standard FIFO queue model
// which has elements in string format. The queue elements in actual SQS
// supports attaching a set of attributes to each elements, but SolidSQS
// generalizes that part and simplifies the element as a string, so use proper
// encoders to send non-string type data.
//
// The biggest problem SolidSQS tries to solve here is to make SQS more
// easier to use in reliable way. AWS API is based on fail-and-retry, once
// data gets stored it's considered reliable but due to what they call
// eventual-consistency and distributed network, it often fails on insertion
// and deletion calls. Dealing with all the exceptions on user-land is not
// something desirable. SolidSQS tries to provide a middle layer that works
// as a reliability controller that helps to minimize user side exception
// handling.
//
// One example, SolidSQS doesn't use batch() series calls. When you push '1',
// '2', '3', if '2' only failed while '1' and '3' succeeded, it usually only
// makes your logic more complicated to fix the failure. This package is focused
// on solving those kinds of hassles. Even SolidSQS provides batch()
// interface, the underlying implementation doesn't use AWS's batch logic except
// of read-only stuff.
//
// For an example, Pop is a transactional operation of peek + delete combo,
// let's say it reads 4 messages, 1~4, and failed at deleting 3, then it will
// return only 1 and 2. And your next call will get the next from there.
//
// Deletion also can be handled in asynchronous manner, in this case it reads
// and returns all then background process will take care of the deletion.
// This could be useful when you need to read all the messages from a queue
// at the fastest possible way in use-and-forget passion.
// It works like a buffered garbage collector. And SQS's visibility feature will
// ensure the data not to be available to other clients until they gets
// deleted.
//
// SolidSQS tries to provide natural way of queue handling that we usually
// think of without bounded by AWS's SQS specific limitations.
// If you're looking for more generic SQS helper implementation, please
// take a look at `common/messagequeue` package.
//
// Note that Amazon SQS makes a best effort to preserve the order of messages,
// but it doesn't guarantee you will receive messages in the exact order you
// sent them. So if preserving order of messages is critical, you should
// consider another solution.
//
// Steve Kim @seungyou 05/10/2016
package solidsqs

import (
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cenkalti/backoff"
	"io"
	"io/ioutil"
	"log"
	"path"
	"strconv"
	"time"
)

const (
	// defaultVisibilityTimeoutSec needs to be just enough to hold the messages
	// for Pop() method to delete after retrieval. Too big value is also no
	// good when batch read performed and returned with partial failure, that
	// failed messages will not be visible again until the timeout.
	defaultVisibilityTimeoutSec = 10
	// maxMessageSize is the maximum size per message. Technically the limit
	// is 255KB including the extra shell part of data, so givning some room
	// for it.
	maxMessageSize = (1024 * 250)
	// maxAsynchronousDeleteLen is maximum number of messages that asynchronous
	// garbage collector can hold. Background deletion works as a buffer so
	// when this queue size is reached, it'll work in synchronous manner.
	maxAsynchronousDeleteLen = 3000

	backoffRandomizationFactor = 0.1
	backoffMultiplier          = 1.5
	defaultRetryIntervalMin    = (100 * time.Millisecond)
	defaultRetryIntervalMax    = (500 * time.Millisecond)
	defaultRetryRuntime        = (3000 * time.Millisecond)
)

// SolidSQS is a class object returned to user.
type SolidSQS struct {
	queue       *sqs.SQS
	queueURL    string
	config      *Config
	logger      *log.Logger
	garbageChan chan sqs.Message
	retryPolicy *backoff.ExponentialBackOff
}

// Config is a sturuct to configure the queue's operational behaviors.
type Config struct {
	// AsynchronousDelete sets background message deletion.
	AsynchronousDelete bool
	// Debug log path. This will dump out all the low level in and out SQS call
	// details.
	DebugOut io.Writer
	// RetryIntervalMin is an initial time interval for the first re-try
	RetryIntervalMin time.Duration
	// RetryIntervalMax caps the growth of interval.
	RetryIntervalMax time.Duration
	// RetryRuntime is the total time elapsed, retry will be stopped when the
	// it exceeds the runtime.
	// it will stop retrying.
	RetryRuntime time.Duration
}

// New creates a new SQS helper
func New(queueURL string, awsConfig *aws.Config, config *Config) (*SolidSQS, error) {
	// Load default configuration if not specified
	if config == nil {
		config = &Config{
			RetryIntervalMin: defaultRetryIntervalMin,
			RetryIntervalMax: defaultRetryIntervalMax,
			RetryRuntime:     defaultRetryRuntime,
		}
	}
	if config.RetryIntervalMin == 0 || config.RetryIntervalMax == 0 || config.RetryRuntime == 0 {
		return nil, fmt.Errorf("Invalid retry configuration parameters")
	}

	// Enable file logging if desired
	logger := log.New(ioutil.Discard, path.Base(queueURL)+" ", log.Ldate|log.Ltime|log.Lmicroseconds)
	if config.DebugOut != nil {
		logger.SetOutput(config.DebugOut)
	}

	// Create a sqs session
	var queue *sqs.SQS
	if awsConfig != nil {
		queue = sqs.New(session.New(awsConfig))
	} else {
		queue = sqs.New(session.New())
	}

	s := &SolidSQS{
		queue:       queue,
		queueURL:    queueURL,
		config:      config,
		logger:      logger,
		garbageChan: nil,
	}

	// Enable garbage collector if desired
	if config.AsynchronousDelete {
		s.garbageChan = make(chan sqs.Message, maxAsynchronousDeleteLen)
		go s.runGarbageCollector()
	}

	logger.Println("A SQS session created.", config)
	return s, nil
}

// Len returns approximate queue length.
// It returns -1 on error rather returning error for easier inline use.
// It can be also used as a ping function.
func (s *SolidSQS) Len() int64 {
	attributes, err := s.info()
	if err != nil {
		s.logger.Println("info failed: ", err)
		return -1
	}

	lenstrp := attributes["ApproximateNumberOfMessages"]
	if lenstrp == nil {
		s.logger.Println("Len() failed", attributes)
		return -1
	}

	var len int64
	len, err = strconv.ParseInt(*lenstrp, 10, 64)
	if err != nil {
		s.logger.Println("Len() failed", attributes)
		return -1
	}

	s.logger.Println("Len()", len)
	return len
}

// Push pushes messages into queue.
//
// It returns number of successful pushed messages along with error.
// For example, if Push("a","b","c","d") returned 2 with error, it failed at "c"
// and all the rest of data after "c" won't be pushed.
//
// - Each message can be up to maxMessageSize (250KB)
// - number of messages can be pushed is unlimited.
func (s *SolidSQS) Push(messages ...string) (int, error) {
	return s.PushWithDelay(0, messages...)
}

// PushWithDelay pushs messages into queue but it'll show up after given
// delay seconds.
func (s *SolidSQS) PushWithDelay(delaySec int, messages ...string) (int, error) {
	if len(messages) == 0 {
		return 0, fmt.Errorf("No messages to push.")
	}
	s.logger.Println("Push()", "messages:", len(messages), "delay:", delaySec)

	numPushed := 0
	for _, message := range messages {
		if err := s.pushMessage(delaySec, message); err != nil {
			return numPushed, err
		}
		numPushed = numPushed + 1
	}
	return numPushed, nil
}

// Pop pops message(s) from the queue and returns them in a string array.
//
// Even though queue have more than requested `maxnum` messages, SQS could
// return less than requested. To prevent, unnecessary user-land loop,
// Pop tries to retrieve all data until maxnum reached or queue gets empty.
// In this case, `pullsec` is only used at the very first internal call.
//
// Pop also eliminates such a limitation of maximum 10 messages per call.
//
// SQS requires popped messages to be deleted with separate call and success
// is not always guaranteed. Push() is ok since the data insertion itself
// fails but the failure of deletion could cause bigger issue because another
// client could pop same messages later. So SolidSQS handles this in 2 ways
//
//  * Synchronous mode: (default)
//   It will try to delete at the time of retrieval, if that fails,
//   it stops and return only the data that deletion has been completed.
//
//  * Asynchronous mode:
//   Deletion doesn't happen at the time of retrieval but happens in
//   background process. This operation could give performance benefit
//   but origianlly designed to cover up temporary deletion fail.
//
// `pullsec` is used for long pull when the queue is empty,
// but it will return immediately if there're available messages.
// `maxnum` is a maximum number of messages, it will try to fetch as manay
// messages as possible that are available at the time.
func (s *SolidSQS) Pop(maxnum int, pullsec int) ([]string, error) {
	exitloop := false
	values := make([]string, 0, maxnum)
	for numPopped := 0; exitloop == false && numPopped < maxnum; {
		// Peek multiple messages from the queue
		resp, err := s.peekMessage(maxnum-numPopped, pullsec, defaultVisibilityTimeoutSec)
		if err != nil || len(resp.Messages) == 0 {
			break
		}

		// Delete the messages from the queue and copy the messages for the return
		for _, message := range resp.Messages {
			if s.config.AsynchronousDelete {
				s.garbageChan <- *message
			} else {
				if err := s.deleteMessage(message); err != nil {
					exitloop = true
					break
				}
			}
			values = append(values, *message.Body)
			numPopped = numPopped + 1
		}

		// From second peek, it needs no wait.
		pullsec = 0
	}

	s.logger.Println("Pop()", "maxnum:", maxnum, "pullsec:", pullsec, "popped:", len(values))
	return values, nil
}

// Purge deletes the messages in the queue.
func (s *SolidSQS) Purge() error {
	return backoff.Retry(func() error {
		_, err := s.queue.PurgeQueue(&sqs.PurgeQueueInput{
			QueueUrl: aws.String(s.queueURL),
		})
		s.logger.Println("Purge()", "error:", err)

		// SQS only allows PurgeQueue operation every 60 seconds.
		// Check if the queue is already empty.
		if err != nil && s.Len() == 0 {
			return nil
		}
		return err
	}, s.newBackoff())
}

// GetAsyncDeleteLen returns running status of asynchronous message deleter.
// It returns remaining number of messages in the pending queue.
func (s *SolidSQS) GetAsyncDeleteLen() int {
	return len(s.garbageChan)
}

func (s *SolidSQS) info() (map[string]*string, error) {
	var attr map[string]*string
	err := backoff.Retry(func() error {
		resp, err := s.queue.GetQueueAttributes(&sqs.GetQueueAttributesInput{
			QueueUrl:       aws.String(s.queueURL),
			AttributeNames: []*string{aws.String("All")},
		})
		if err == nil {
			attr = resp.Attributes
		}
		return err
	}, s.newBackoff())
	return attr, err
}

func (s *SolidSQS) pushMessage(delaySec int, message string) error {
	if len(message) > maxMessageSize {
		return fmt.Errorf("Message is too big. must %d < %d", len(message), maxMessageSize)
	}

	return backoff.Retry(func() error {
		_, err := s.queue.SendMessage(&sqs.SendMessageInput{
			QueueUrl:     aws.String(s.queueURL),
			MessageBody:  aws.String(message),
			DelaySeconds: aws.Int64(int64(delaySec)),
		})
		s.logger.Println("pushMessage()", "error:", err, "delay:", delaySec, "-", message)
		return err
	}, s.newBackoff())
}

func (s *SolidSQS) peekMessage(maxnum int, pullsec int, vtimeout int) (*sqs.ReceiveMessageOutput, error) {
	var resp *sqs.ReceiveMessageOutput
	err := backoff.Retry(func() error {
		var err error
		resp, err = s.queue.ReceiveMessage(&sqs.ReceiveMessageInput{
			QueueUrl:            aws.String(s.queueURL),
			MaxNumberOfMessages: aws.Int64(1),
			VisibilityTimeout:   aws.Int64(int64(vtimeout)),
			WaitTimeSeconds:     aws.Int64(int64(pullsec)),
		})
		s.logger.Println("peekMessage()", "maxnum:", maxnum, "pullsec:", pullsec, "vtimeout:", vtimeout, "error:", err, "-", resp)
		return err
	}, s.newBackoff())
	return resp, err
}

func (s *SolidSQS) deleteMessage(message *sqs.Message) error {
	return backoff.Retry(func() error {
		_, err := s.queue.DeleteMessage(&sqs.DeleteMessageInput{
			QueueUrl:      aws.String(s.queueURL),
			ReceiptHandle: message.ReceiptHandle,
		})
		s.logger.Println("deleteMessage()", "error:", err, "-", *message.ReceiptHandle)
		return err
	}, s.newBackoff())
}

func (s *SolidSQS) runGarbageCollector() {
	for {
		select {
		case message := <-s.garbageChan:
			// TODO: Switch to batch deletion and Add delayed retry option
			err := s.deleteMessage(&message)
			s.logger.Println("runGarbageCollector()", "error:", err, "-", message)
		}
	}
}

func (s *SolidSQS) newBackoff() *backoff.ExponentialBackOff {
	b := backoff.NewExponentialBackOff()
	b.InitialInterval = s.config.RetryIntervalMin
	b.RandomizationFactor = backoffRandomizationFactor
	b.Multiplier = backoffMultiplier
	b.MaxInterval = s.config.RetryIntervalMax
	b.MaxElapsedTime = s.config.RetryRuntime
	b.Reset()
	return b
}
