// Package spadefakes provides a fake spade.Client for testing.
package spadefakes

import (
	"bytes"
	"context"
	"encoding/gob"
	"log"
	"sync"

	"code.justin.tv/common/spade-client-go/spade"
)

type trackEventArgs struct {
	ctx        context.Context
	event      string
	properties interface{}
}

// FakeClient implements the spade.Client interface and allows testing code to
// see invocations of TrackEvent() and TrackEvents() and their results.
type FakeClient struct {
	trackEventMutex   sync.RWMutex
	trackEventStub    func(ctx context.Context, event string, properties interface{}) error
	trackEventReturns struct {
		result1 error
	}
	trackEventArgsForCall []trackEventArgs

	trackEventsMutex       sync.RWMutex
	trackEventsStub        func(ctx context.Context, events ...spade.Event) error
	trackEventsReturns     error
	trackEventsArgsForCall []struct {
		ctx    context.Context
		events []spade.Event
	}

	invocations      map[string][][]interface{}
	invocationsMutex sync.RWMutex
}

// TrackEvent records the given parameters, then passes them to trackEventStub and returns the result.
// If trackEventStub is missing, it returns the error specified in trackEventReturns.result1 if any.
func (fake *FakeClient) TrackEvent(ctx context.Context, event string, properties interface{}) error {
	fake.recordInvocation("TrackEvent", []interface{}{ctx, event, properties})

	fake.trackEventMutex.Lock()
	defer fake.trackEventMutex.Unlock()

	fake.trackEventArgsForCall = append(fake.trackEventArgsForCall, trackEventArgs{ctx, event, properties})
	if fake.trackEventStub != nil {
		return fake.trackEventStub(ctx, event, properties)
	}
	return fake.trackEventReturns.result1
}

// TrackEventCallCount returns the number of times TrackEvent() has been called on the FakeClient.
func (fake *FakeClient) TrackEventCallCount() int {
	fake.trackEventMutex.RLock()
	defer fake.trackEventMutex.RUnlock()

	return len(fake.trackEventArgsForCall)
}

// TrackEventArgsForCall returns the arguments used in the given call of TrackEvents.
func (fake *FakeClient) TrackEventArgsForCall(i int) (context.Context, string, interface{}) {
	fake.trackEventMutex.RLock()
	defer fake.trackEventMutex.RUnlock()

	args := fake.trackEventArgsForCall[i]
	return args.ctx, args.event, args.properties
}

// SetTrackEventStub sets trackEventStub in a thread-safe manner.
func (fake *FakeClient) SetTrackEventStub(f func(ctx context.Context, event string, properties interface{}) error) {
	fake.trackEventMutex.Lock()
	defer fake.trackEventMutex.Unlock()

	fake.trackEventStub = f
}

// TrackEventReturns removes trackEventStub if any and sets the returned error
// on future invocations of TrackEvent.
func (fake *FakeClient) TrackEventReturns(result1 error) {
	fake.trackEventMutex.Lock()
	defer fake.trackEventMutex.Unlock()

	fake.trackEventStub = nil
	fake.trackEventReturns = struct{ result1 error }{result1}
}

// TrackEvents records the given parameters, then passes them to trackEventsStub and returns the result.
// If trackEventsStub is missing, it returns the error specified in trackEventsReturns if any.
func (fake *FakeClient) TrackEvents(ctx context.Context, events ...spade.Event) error {
	fake.recordInvocation("TrackEvents", []interface{}{ctx, events})

	fake.trackEventsMutex.Lock()
	defer fake.trackEventsMutex.Unlock()

	fake.trackEventsArgsForCall = append(fake.trackEventsArgsForCall, struct {
		ctx    context.Context
		events []spade.Event
	}{ctx, events})

	if fake.trackEventsStub != nil {
		return fake.trackEventsStub(ctx, events...)
	}
	return fake.trackEventsReturns
}

// TrackEventsCallCount returns the number of times TrackEvents() has been called on the FakeClient.
func (fake *FakeClient) TrackEventsCallCount() int {
	fake.trackEventsMutex.RLock()
	defer fake.trackEventsMutex.RUnlock()

	return len(fake.trackEventsArgsForCall)
}

// TrackEventsArgsForCall returns the arguments used in the given call of TrackEvents.
func (fake *FakeClient) TrackEventsArgsForCall(i int) (context.Context, []spade.Event) {
	fake.trackEventsMutex.RLock()
	defer fake.trackEventsMutex.RUnlock()

	args := fake.trackEventsArgsForCall[i]
	return args.ctx, args.events
}

// SetTrackEventsStub sets trackEventsStub in a thread-safe manner.
func (fake *FakeClient) SetTrackEventsStub(f func(ctx context.Context, events ...spade.Event) error) {
	fake.trackEventsMutex.Lock()
	defer fake.trackEventsMutex.Unlock()

	fake.trackEventsStub = f
}

// TrackEventsReturns removes the trackEventsStub if any and sets the returned error
// on future invocations of TrackEvents.
func (fake *FakeClient) TrackEventsReturns(result1 error) {
	fake.trackEventsMutex.Lock()
	defer fake.trackEventsMutex.Unlock()

	fake.trackEventsStub = nil
	fake.trackEventsReturns = result1
}

// Invocations returns the map of TrackEvent and TrackEvents so far.  We use gob to return a
// deep copy of the map, since access to the returned map is not protected by a lock.
func (fake *FakeClient) Invocations() map[string][][]interface{} {
	fake.invocationsMutex.RLock()
	defer fake.invocationsMutex.RUnlock()

	var buffer bytes.Buffer
	if err := gob.NewEncoder(&buffer).Encode(fake.invocations); err != nil {
		log.Fatal("failed to encode invocations")
	}
	var copy map[string][][]interface{}
	if err := gob.NewDecoder(&buffer).Decode(&copy); err != nil {
		log.Fatal("failed to decode invocations")
	}
	return copy
}

func (fake *FakeClient) recordInvocation(key string, args []interface{}) {
	fake.invocationsMutex.Lock()
	defer fake.invocationsMutex.Unlock()

	if fake.invocations == nil {
		fake.invocations = map[string][][]interface{}{}
	}
	if fake.invocations[key] == nil {
		fake.invocations[key] = [][]interface{}{}
	}
	fake.invocations[key] = append(fake.invocations[key], args)
}

var _ spade.Client = new(FakeClient)
