// Package rtevent provides a set of APIs to send run-time events.
package rtevent

import (
	"code.justin.tv/common/solidsqs"
	"encoding/json"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"io"
	"net"
	"time"
)

const (
	sqsURL              = "https://sqs.us-west-2.amazonaws.com/043714768218/skadi_rtevent_prod"
	sqsRegion           = "us-west-2"
	sqsRetryIntervalMin = (100 * time.Millisecond)
	sqsRetryIntervalMax = (500 * time.Millisecond)
	sqsRetryRuntime     = (3000 * time.Millisecond)
)

// Client is the object of RtEvent
type Client struct {
	queue     *solidsqs.SolidSQS
	publisher bool
}

// NewClient creates a new event client.
// This client is for the event receiver applications. Use NewPublisherClient() for user applications.
func NewClient(awsConfig *aws.Config, out io.Writer) (*Client, error) {
	if awsConfig == nil {
		awsConfig = &aws.Config{Region: aws.String(sqsRegion)}
	}

	// Create a new SQS object
	sqsQueue, err := solidsqs.New(sqsURL, awsConfig, &solidsqs.Config{
		RetryIntervalMin:   sqsRetryIntervalMin,
		RetryIntervalMax:   sqsRetryIntervalMax,
		RetryRuntime:       sqsRetryRuntime,
		AsynchronousDelete: false,
		DebugOut:           out,
	})
	if err != nil {
		return nil, err
	}

	s := &Client{
		queue: sqsQueue,
	}

	return s, nil
}

// NewPublisherClient creates a new event client with limited access only for event publishing.
// All the end users need to get this client.
func NewPublisherClient() (*Client, error) {
	c, err := NewClient(nil, nil)
	if err == nil && c != nil {
		c.publisher = true
	}
	return c, err
}

// SendDeployEvent sends a deployment event.
func (c *Client) SendDeployEvent(event *DeployEvent) error {
	err := checkAndCompleteDeployEvent(event)
	if err != nil {
		return err
	}

	jEvent, err := json.Marshal(event)
	if err != nil {
		return fmt.Errorf("%v - %v", err, *event)
	}

	_, err = c.queue.Push(string(jEvent))
	return err
}

// SendDeployEvent sends a deployment event on behalf of
func (c *Client) SendDeployEventOnbehalf(event *DeployEvent, remoteAddr string) error {
	err := checkAndCompleteDeployEvent(event)
	if err != nil {
		return err
	}

	// Update remote host info
	if net.ParseIP(remoteAddr) != nil {
		// remoteAddr is IP address
		event.Hostname = remoteAddr // fill IP address for hostname
		event.IPAddr = remoteAddr
	} else {
		// remoteAddr is hostname
		event.Hostname = remoteAddr
		ips, err := net.LookupIP(remoteAddr)
		if err != nil {
			return err
		}
		if len(ips) == 0 {
			return fmt.Errorf("Unexpected error: lookup succeeded without returning IP address for %v", remoteAddr)
		}
		event.IPAddr = ips[0].String()
	}

	jEvent, err := json.Marshal(event)
	if err != nil {
		return fmt.Errorf("%v - %v", err, *event)
	}

	_, err = c.queue.Push(string(jEvent))
	return err
}

// PullEvent pulls event from the queue.
func (c *Client) PullEvent(maxnum int, pullsec int) ([]string, error) {
	if c.publisher {
		return nil, fmt.Errorf("Not supported")
	}
	return c.queue.Pop(maxnum, pullsec)
}
