package rtevent

import (
	"fmt"
	"os"
	"strings"
	"time"
)

const (
	// EventMessageVersion is a version of message format.
	EventMessageVersion = 1

	// EventTypeHeader type for header only event.
	EventTypeHeader = 0
	// EventTypeDeploy type for deployment event.
	EventTypeDeploy = 1
)

// EventHeader defines common fields across all the event messages.
type EventHeader struct {
	Version   int8      `json:"version"`
	Type      int8      `json:"type"`
	IPAddr    string    `json:"ipaddr"`
	Hostname  string    `json:"hostname"`
	CreatedAt time.Time `json:"createdat"`
}

// DeployEvent defines a message for deployment event.
type DeployEvent struct {
	EventHeader // Header, Auto generated by API.

	// User input
	App         string `json:"app"`         // application name ex) dta/rtevent
	Sha         string `json:"sha"`         // sha version string
	Environment string `json:"environment"` // environment name
	Success     bool   `json:"success"`     // success or fail

	Phase    string `json:"phase,omitempty"`    // deploy phase (optional)
	Deployer string `json:"deployer,omitempty"` // deployer name (optional)
	DeployID uint64 `json:"deployid,omitempty"` // deploy identifier (optional)
	Desc     string `json:"desc,omitempty"`     // extra information (optional)
}

func generateEventHeader(event interface{}) error {
	ipAddr, err := getLocalIP()
	if err != nil {
		return err
	}

	hostname, err := os.Hostname()
	if err != nil {
		return err
	}

	header := EventHeader{
		Version:   EventMessageVersion,
		Type:      EventTypeHeader,
		IPAddr:    ipAddr,
		Hostname:  hostname,
		CreatedAt: time.Now(),
	}

	if e, ok := event.(*EventHeader); ok {
		*e = header
		return nil
	}
	if e, ok := event.(*DeployEvent); ok {
		header.Type = EventTypeDeploy
		e.EventHeader = header
		return nil
	}

	return fmt.Errorf("Invalid argument - failed to determine event message type")

}

func checkAndCompleteDeployEvent(event *DeployEvent) error {
	event.App = strings.TrimSpace(event.App)
	event.Sha = strings.TrimSpace(event.Sha)
	event.Environment = strings.TrimSpace(event.Environment)
	if event.App == "" || event.Sha == "" {
		return fmt.Errorf("Invalid arguments - App and Sha fields must be provided")
	}

	if err := generateEventHeader(event); err != nil {
		return err
	}

	return nil
}
