package api

import "fmt"

type Settings struct {
	Deploy *DeployConfig `json:"deploy"`
}

type DeployConfig struct {
	Artifact       *string   `json:"artifact,omitempty"`
	Job            *string   `json:"job"`
	ConsulServices *[]string `json:"consul_services"`
	// List of user-defined datacenters for:
	//   - where consul search must be performed
	//   - will be used in per-dc deployment (not supported yet as of 4/1/2017)
	Datacenters        *[]string                        `json:"datacenters"`
	Environments       *map[string]AppEnvironmentConfig `json:"environments"`
	Distribution       *DeployDistributionConfig        `json:"distribution,omitempty"`
	Restart            *DeployRestartConfig             `json:"restart,omitempty"`
	GraphiteTargets    *[]string                        `json:"graphite_targets"`
	Slack              *ChatConfig                      `json:"slack,omitempty"`
	Smoca              *SmocaConfig                     `json:"smoca,omitempty"`
	BuildStatusContext *string                          `json:"build_status_context,omitempty"`
	RequiredContexts   *[]string                        `json:"required_contexts,omitempty"`
}

// AppEnvironmentConfig allows repos to have separate jenkins deploy jobs per environment
type AppEnvironmentConfig struct {
	Job              *string                                 `json:"job"`
	Subscriptions    map[string]NotificationSubscriberConfig `json:"subscriptions,omitempty"`
	WhiteList        *[]string                               `json:"whitelist,omitempty"`
	RequiredContexts *[]string                               `json:"required_contexts,omitempty"`
	AWSAccountID     *string                                 `json:"awsaccountid,omitempty"`
	Artifact         *string                                 `json:"artifact,omitempty"`
}

type SmocaConfig struct {
	Hidden *bool `json:"hidden,omitempty"`
	OptOut *bool `json:"opt_out,omitempty"`
}

// IsBranchWhiteListed returns true if the branch is whitelisted for the environment or not
// Default to true if there is no whitelist specified for the environment.
func (c *DeployConfig) IsBranchWhiteListed(env string, branch string) (bool, error) {

	envs := c.Environments

	if envs != nil {
		appEnv, ok := (*envs)[env]
		if ok {
			if (appEnv.WhiteList != nil) && (len(*appEnv.WhiteList) > 0) {
				for _, wlitem := range *appEnv.WhiteList {
					if branch == wlitem {
						return true, nil
					}
				}
				return false, fmt.Errorf("Environment %v has a whitelist which does not include the current branch %v", env, branch)
			}
		}
	}
	return true, nil
}

// EnvironmentRequiredContexts returns a list of contexts that must be successful for a deployment to work.
// It uses the `required_contexts` config from the given environment, or if there isn't one,
// a global `required_contexts`.
// If both are undefined, a global `build_status_context` will be used. This will be deprecated in the next major release.
// An empty list will make all deployments succeed.
// If no config is available then all contexts must succeed.
func (c *DeployConfig) EnvironmentRequiredContexts(env string) *[]string {
	contexts := []string{}
	if c.Environments != nil {
		appEnv, ok := (*c.Environments)[env]
		if ok && appEnv.RequiredContexts != nil {
			for _, ctx := range *appEnv.RequiredContexts {
				contexts = append(contexts, ctx)
			}
			return &contexts
		}
	}
	if c.RequiredContexts != nil {
		for _, ctx := range *c.RequiredContexts {
			contexts = append(contexts, ctx)
		}
		return &contexts
	}
	if c.BuildStatusContext != nil {
		return &[]string{*c.BuildStatusContext}
	}
	return nil
}

type NotificationSubscriberConfig struct {
	Success bool `json:"success"`
	Failure bool `json:"failure"`
	Pending bool `json:"pending"`
}

// GetDeployJob returns the appropriate jenkins job for deploying the given environment.  Environment-specific
// settings override the global job definition
func (c *DeployConfig) GetDeployJob(env string) (string, error) {
	var job string

	envs := c.Environments

	if envs != nil {
		appEnv, ok := (*envs)[env]
		if ok {
			if appEnv.Job != nil {
				job = *appEnv.Job
			}
		}
	}

	if job == "" && c.Job != nil {
		job = *c.Job
	}

	if job == "" {
		return "", fmt.Errorf("no deploy job defined for environment:%v globally or in environments:%+v", env, envs)
	}

	return job, nil
}

// GetDeployArtifact returns the appropriate artifact type for deploying the given environment.  Environment-specific
// settings override the global artifact definition
func (c *DeployConfig) GetDeployArtifact(env string) (string, error) {
	if c.Environments != nil {
		appEnv, ok := (*c.Environments)[env]
		if ok {
			if appEnv.Artifact != nil {
				return *appEnv.Artifact, nil
			}
		}
	}
	if c.Artifact != nil {
		return *c.Artifact, nil
	}
	return "", fmt.Errorf("no artifact defined for environment:%v globally or in environments:%+v", env, c.Environments)
}

type DeployDistributionConfig struct {
	Style             *string  `json:"style"`
	Concurrency       *uint    `json:"concurrency,omitempty"`
	FailThresholdType *string  `json:"fail_threshold_type,omitempty"`
	FailThreshold     *float32 `json:"fail_threshold,omitempty"`
}

type DeployRestartConfig struct {
	Style             *string  `json:"style"`
	Concurrency       *uint    `json:"concurrency,omitempty"`
	Service           *string  `json:"service,omitempty"`
	Signal            *string  `json:"signal,omitempty"`
	Wait              *uint    `json:"wait,omitempty"`
	Uptime            *uint    `json:"uptime,omitempty"`
	FailThresholdType *string  `json:"fail_threshold_type,omitempty"`
	FailThreshold     *float32 `json:"fail_threshold,omitempty"`
}

type Repositories struct {
	c *Client
}

func (c *Client) Repositories() *Repositories {
	return &Repositories{c}
}

func (r *Repositories) Settings(owner, name, ref string) (*Settings, error) {
	var settings Settings
	req, err := r.c.NewRequest("GET", fmt.Sprintf("/v1/repos/%v/%v/settings?ref=%v", owner, name, ref), nil)
	if err != nil {
		return nil, err
	}

	_, err = r.c.Do(req, &settings)
	if err != nil {
		return nil, err
	}

	return &settings, nil
}
