package api

import (
	"fmt"
	"net/http"
	"testing"
)

func TestSettings(t *testing.T) {
	setup()
	defer teardown()

	mux.HandleFunc("/v1/repos/release/test/settings", func(w http.ResponseWriter, r *http.Request) {
		testMethod(t, r, "GET")
		fmt.Fprint(w, `{"deploy": {"restart": {"style": "fast"}}}`)
	})

	settings, err := client.Repositories().Settings("release", "test", "master")
	if err != nil {
		t.Fatalf("Repositories.Settings returned err: %v", err)
	}

	if *settings.Deploy.Restart.Style != "fast" {
		t.Fatalf("Repositories.Settings did not parse")
	}
}

func TestGradualRestart(t *testing.T) {
	setup()
	defer teardown()

	mux.HandleFunc("/v1/repos/release/test/settings", func(w http.ResponseWriter, r *http.Request) {
		testMethod(t, r, "GET")
		fmt.Fprint(w, `{"deploy": {"restart": {"style": "gradual", "concurrency": 4}}}`)
	})

	settings, err := client.Repositories().Settings("release", "test", "master")
	if err != nil {
		t.Fatalf("Repositories.Settings returned err: %v", err)
	}

	if *settings.Deploy.Restart.Style != "gradual" {
		t.Fatalf("Repositories.Settings did not parse")
	}

	if *settings.Deploy.Restart.Concurrency != 4 {
		t.Fatalf("Respositories.Settings did not parse")
	}
}

func TestService(t *testing.T) {
	setup()
	defer teardown()

	mux.HandleFunc("/v1/repos/release/test/settings", func(w http.ResponseWriter, r *http.Request) {
		testMethod(t, r, "GET")
		fmt.Fprint(w, `{"deploy": {"restart": {"style": "gradual", "concurrency": 1, "service": "*jax*"}}}`)
	})

	settings, err := client.Repositories().Settings("release", "test", "master")
	if err != nil {
		t.Fatalf("Repositories.Settings returned err: %v", err)
	}

	if *settings.Deploy.Restart.Service != "*jax*" {
		t.Fatalf("Repositories.Settings did not parse")
	}
}

func TestGetDeployJob(t *testing.T) {
	job1 := "foo"
	c := &DeployConfig{
		Job: &job1,
	}

	job, _ := c.GetDeployJob("env")
	if job != job1 {
		t.Errorf("expected job to be %v but got %v", job1, job)
	}
}

func TestGetDeployJobFromEnv(t *testing.T) {
	job1 := "foo"
	job2 := "bar"
	envs := make(map[string]AppEnvironmentConfig)
	envs["env"] = AppEnvironmentConfig{
		Job: &job2,
	}
	c := &DeployConfig{
		Job:          &job1,
		Environments: &envs,
	}

	job, _ := c.GetDeployJob("env")
	if job != job2 {
		t.Errorf("expected job to be %v but got %v", job2, job)
	}
}

func TestGetArtifact(t *testing.T) {
	artifact1 := "tar"
	c := &DeployConfig{
		Artifact: &artifact1,
	}
	artifact, _ := c.GetDeployArtifact("env")
	if artifact != artifact1 {
		t.Errorf("expected artifact to be %v but got %v", artifact1, artifact)
	}
}

func TestGetArtifactEmpty(t *testing.T) {
	c := &DeployConfig{}
	artifact, _ := c.GetDeployArtifact("env")
	if artifact != "" {
		t.Errorf("expected artifact to be empty string but got %v", artifact)
	}
}

func TestGetArtifactFromEnv(t *testing.T) {
	artifact1 := "tar"
	artifact2 := "pkg"
	c := &DeployConfig{
		Artifact: &artifact1,
		Environments: &map[string]AppEnvironmentConfig{
			"env": AppEnvironmentConfig{
				Artifact: &artifact2,
			},
		},
	}
	artifact, _ := c.GetDeployArtifact("env")
	if artifact != artifact2 {
		t.Errorf("expected artifact to be %v but got %v", artifact2, artifact)
	}
}

func TestGetArtifactFromOtherEnv(t *testing.T) {
	artifact1 := "tar"
	artifact2 := "pkg"
	c := &DeployConfig{
		Artifact: &artifact1,
		Environments: &map[string]AppEnvironmentConfig{
			"env": AppEnvironmentConfig{
				Artifact: &artifact2,
			},
		},
	}
	artifact, _ := c.GetDeployArtifact("otherEnv")
	if artifact != artifact1 {
		t.Errorf("expected artifact to be %v but got %v", artifact2, artifact)
	}
}

func TestNilEnvironmentJob(t *testing.T) {
	job := "foo"
	envs := make(map[string]AppEnvironmentConfig)
	envs["production"] = AppEnvironmentConfig{}
	c := &DeployConfig{
		Job:          &job,
		Environments: &envs,
	}

	actual, _ := c.GetDeployJob("production")
	if actual != job {
		t.Errorf("expected job to %v but got %v", job, actual)
	}
}

func TestNilRootJob(t *testing.T) {
	job := "foo"
	envs := make(map[string]AppEnvironmentConfig)
	envs["production"] = AppEnvironmentConfig{
		Job: &job,
	}
	c := &DeployConfig{
		Environments: &envs,
	}

	actual, _ := c.GetDeployJob("production")
	if actual != job {
		t.Errorf("expected job to %v but got %v", job, actual)
	}
}

func TestNilJob(t *testing.T) {
	envs := make(map[string]AppEnvironmentConfig)
	envs["staging2"] = AppEnvironmentConfig{}
	c := &DeployConfig{
		Environments: &envs,
	}

	_, err := c.GetDeployJob("staging2")
	if err == nil {
		t.Errorf("expected method to error, did not")
	}
}

func TestEnvironmentRequiredContext(t *testing.T) {
	testEnv := "testEnv"
	buildContext := "test-context"
	globalContexts := []string{"globalCtx1", "globalCtx2", "globalCtx3"}
	envContexts := []string{"envCtx1", "envCtx2"}
	emptyContexts := []string{}
	tests := []struct {
		name     string
		dc       DeployConfig
		expected *[]string
		errorMsg string
	}{
		{
			name: "unconfigured",
			dc: DeployConfig{
				Environments:       nil,
				RequiredContexts:   nil,
				BuildStatusContext: nil,
			},
			expected: nil,
			errorMsg: "Expected nil to pass to required contexts",
		}, {
			name: "build status context set",
			dc: DeployConfig{
				Environments:       nil,
				RequiredContexts:   nil,
				BuildStatusContext: &buildContext,
			},
			expected: &[]string{buildContext},
			errorMsg: "build status context didn't work",
		}, {
			name: "global contexts set",
			dc: DeployConfig{
				Environments:       nil,
				RequiredContexts:   &globalContexts,
				BuildStatusContext: nil,
			},
			expected: &globalContexts,
			errorMsg: "failed global contexts",
		}, {
			name: "non matching environment",
			dc: DeployConfig{
				Environments:       &map[string]AppEnvironmentConfig{"anotherEnv": {RequiredContexts: &envContexts}},
				RequiredContexts:   nil,
				BuildStatusContext: nil,
			},
			expected: nil,
			errorMsg: "Expected nil when no env matches",
		}, {
			name: "matching environment",
			dc: DeployConfig{
				Environments:       &map[string]AppEnvironmentConfig{testEnv: {RequiredContexts: &envContexts}},
				RequiredContexts:   nil,
				BuildStatusContext: nil,
			},
			expected: &envContexts,
			errorMsg: "failed environment contexts",
		}, {
			name: "matched environment is empty",
			dc: DeployConfig{
				Environments:       &map[string]AppEnvironmentConfig{testEnv: {RequiredContexts: &emptyContexts}},
				RequiredContexts:   nil,
				BuildStatusContext: nil,
			},
			expected: &emptyContexts,
			errorMsg: "Expected empty contexts for environment",
		}, {
			name: "global contexts empty",
			dc: DeployConfig{
				Environments:       nil,
				RequiredContexts:   &emptyContexts,
				BuildStatusContext: nil,
			},
			expected: &emptyContexts,
			errorMsg: "Expected empty global contexts",
		}, {
			name: "environment contexts have higher priority than global",
			dc: DeployConfig{
				Environments:       &map[string]AppEnvironmentConfig{testEnv: {RequiredContexts: &envContexts}},
				RequiredContexts:   &globalContexts,
				BuildStatusContext: nil,
			},
			expected: &envContexts,
			errorMsg: "environment contexts aren't higher priority than global context",
		}, {
			name: "global contexts have higher priority than build context",
			dc: DeployConfig{
				Environments:       nil,
				RequiredContexts:   &globalContexts,
				BuildStatusContext: &buildContext,
			},
			expected: &globalContexts,
			errorMsg: "global context isn't higher priority than build context",
		}, {
			name: "environment contexts has highest priority",
			dc: DeployConfig{
				Environments:       &map[string]AppEnvironmentConfig{testEnv: {RequiredContexts: &envContexts}},
				RequiredContexts:   &globalContexts,
				BuildStatusContext: &buildContext,
			},
			expected: &envContexts,
			errorMsg: "environment contexts aren't higher priority than global contexts and build context",
		},
	}
	for _, test := range tests {
		t.Run(test.name, func(t *testing.T) {
			res := test.dc.EnvironmentRequiredContexts(testEnv)
			if test.expected == nil {
				if res != nil {
					t.Error(test.errorMsg)
				}
			} else {
				if len(*res) != len(*test.expected) {
					t.Errorf("%s: Unexpected number of contexts, expected %d but got %d", test.errorMsg, len(*test.expected), len(*res))
				}
				for i, ctx := range *test.expected {
					if ctx != (*res)[i] {
						t.Errorf("%s: Expected context #%d didn't match. Expected `%s` but got `%s`", test.errorMsg, i, ctx, (*res)[i])
					}
				}
			}
		})
	}
}
