app.service('deployments', ['$q', '$http', '$interval', '$cookies', '$state', 'config', 'repos', function($q, $http, $interval, $cookies, $state, config, repos) {

	this.environments = []
	this.deploys = {}
	this.statuses = {}
	this.watched_deploys = []
	this.watchers = {}
	this.logs = {}
	this.watched_logs = {}
	this.events = {}
	this.timeSinceLastDeploy = {}

	var that = this

	this.set_repo = function(owner, repo) {
		var deferred = $q.defer()

		if (owner == that.owner && repo == that.repo) {
			deferred.resolve()
			return deferred.promise
		}

		that.set_environments(owner, repo)

		that.load_recent_deploys()
			.then(that.load_deployment_statuses)
			.then(function() {
				deferred.resolve()
			})

		return deferred.promise
	}

	this.set_environments = function(owner, repo) {
		that.owner = owner
		that.repo = repo
		if (repos.settings != null && repos.settings.deploy != null) {
			that.environments = Object.keys(repos.settings.deploy.environments)
		}
	}

	this.create = function(branch, ref, environment, description, codeReviewURL, riskAssessment, triggerSmoca, hosts, link) {
		data = $.param({
			owner: that.owner,
			repo: that.repo,
			ref: ref,
			branch: branch,
			environment: environment,
			description: description,
			codeReviewURL: codeReviewURL,
			severity: riskAssessment,
			triggerSmoca: triggerSmoca,
			hosts: hosts,
			link: link
		})

		c = {
			headers: {
				'Content-Type': 'application/x-www-form-urlencoded'
			}
		}

		return $http.post(config.skadi_host+"/v1/deployments", data, c)
			.success(function(deployment) {
				if (that.deploys[deployment.environment] === undefined) { that.deploys[deployment.environment] = []}
				that.deploys[deployment.environment].unshift(deployment)
				that.start_watching(deployment)
				$state.go('repo.deploy', {id: deployment.id})
			})
	}

	this.get = function(id) {
		var deferred = $q.defer();
		var found = false

		for (var env in that.deploys) {
			that.deploys[env].forEach(function(deploy) {
				if (deploy.id == id) {
					found = true
					deferred.resolve(deploy)
				}
			})
		}

		if (!found) {
			that.get_deployment(id).then(function(deploy){
				deferred.resolve(deploy)
			})
		}

		return deferred.promise
	}

	// start_watching starts an $interval which queries the status of a deployment until it either succeeds
	// or fails.
	this.start_watching = function(deploy) {
		if (that.watched_deploys.indexOf(deploy.id) == -1) {
			that.watched_deploys.push(deploy.id)

			that.watchers[deploy.id] = $interval(function() {
				if (that.watched_logs[deploy.id] === true) {
					that.get_log_output(deploy.id)
						.then(function(log) {
							that.logs[deploy.id] = log
						})
				}
				that.load_deployment_status_for_deploy(deploy, false)
					.then(function() {
						if (that.statuses[deploy.id].state == "success" ||
							that.statuses[deploy.id].state == "failure" ||
							that.statuses[deploy.id].state == "error" ||
							that.statuses[deploy.id].state == "aborted") {
							if ($interval.cancel(that.watchers[deploy.id])) {
								that.watched_logs[deploy.id] = false
								that.watched_deploys.splice(that.watched_deploys.indexOf(deploy.id), 1)
								delete that.watchers[deploy.id]
							}
						}
					})
			}, 10000)
		}
	}

	this.load_recent_deploys = function() {
		var promises = []

		that.environments.forEach(function(environment) {
			p = that.get_deploys_for_env(environment)
				.then(function(deployments) {
					var env = environment
					that.deploys[env] = deployments
				})

			promises.push(p)
		})

		return $q.all(promises)
	}

	this.set_recent_deploys_for_env = function(environment, deploys_per_env) {
		var deferred = $q.defer()

		that.get_deploys_for_env(environment, deploys_per_env)
			.then(function(deployments) {
				var env = environment
				that.deploys[env] = deployments
				that.deploys[env].forEach(function(deploy) {
					that.load_deployment_status_for_deploy(deploy, true)
					deferred.resolve()
				})
			})

		return deferred.promise
	}

	this.set_recent_deploy_events_for_env = function(environment, deploys_per_env) {
		var deferred = $q.defer()

		that.get_deploy_events_for_env(environment, deploys_per_env)
			.then(function(events) {
				var env = environment
				var lastDeploy = null
				that.events[env] = events
				events.forEach(function(evnt) {
					if (evnt.event_type === "deployment") {
						var deploy = evnt.deployment
						if (lastDeploy) {
							that.timeSinceLastDeploy[lastDeploy.id] = moment(deploy.created_at).from(moment(lastDeploy.created_at), true)
						}
						that.load_deployment_status_for_deploy(deploy, true)
						lastDeploy = deploy
					}
					deferred.resolve()
				})
			})
		return deferred.promise
	}

	this.get_deploys_for_env = function(environment, count) {
		var deferred = $q.defer()

		// default to 1 deployment to help with performance
		count = typeof count !== 'undefined' ? count : '1';

		$http.get(config.skadi_host+"/v1/deployments?owner="+that.owner+"&repo="+that.repo+"&environment="+encodeURIComponent(environment)+"&count="+count)
			.success(function(deployments) {
				deferred.resolve(deployments)
			})

		return deferred.promise
	}

	this.get_deploy_events_for_env = function(environment, count) {
		var deferred = $q.defer()

		$http.get(config.skadi_host+"/v1/deployments/history?owner="+that.owner+"&repo="+that.repo+"&environment="+encodeURIComponent(environment)+"&count="+count)
			.success(function(events) {
				deferred.resolve(events)
			})

		return deferred.promise
	}

	this.load_deployment_statuses = function() {
		var promises = []

		that.environments.forEach(function(environment) {
			var env = environment

			that.deploys[env].forEach(function(deploy) {
				p = that.load_deployment_status_for_deploy(deploy, true)
				promises.push(p)
			})
		})

		return $q.all(promises)
	}

	this.load_deployment_status_for_deploy = function(deploy, watch) {
		return $http.get(config.skadi_host+"/v1/deployments/"+deploy.id+"/status")
			.success(function(status) {
				that.statuses[deploy.id] = status
				that.add_console_url(deploy)
				if (watch) {
					if (status.state == "pending" || status.state == "unknown") {
						that.start_watching(deploy)
					}
				}
			})
	}

	this.add_console_url = function(deploy) {
		if (that.statuses[deploy.id].target_url) {
			that.statuses[deploy.id].console_url = that.statuses[deploy.id].target_url + 'consoleFull'
			that.statuses[deploy.id].params_url = that.statuses[deploy.id].target_url + 'parameters'
		}
	}

	this.deploy_percentage = function(deploy_id) {
		if (deploy_id && that.statuses[deploy_id]) {
			return that.statuses[deploy_id].percent
		}

		return 0
	}

	this.get_log_output = function(deploy_id) {
		var deferred = $q.defer()

		$http.get(config.skadi_host+"/v1/deployments/"+deploy_id+"/log")
			.success(function(log) {
				deferred.resolve(log.output)
			})

		return deferred.promise
	}

	this.get_deployment = function(deploy_id) {
		var deferred = $q.defer()

		$http.get(config.skadi_host+"/v1/deployment/"+deploy_id)
			.success(function(deploy){
				deferred.resolve(deploy)
			})

		return deferred.promise
	}
}]);
