-- +goose Up
-- SQL in section 'Up' is executed when this migration is applied
CREATE TABLE environments(
	Environment    TEXT NOT NULL,
	Owner          TEXT NOT NULL,
	Repository     TEXT NOT NULL,
	DeployedSHA    CHAR(40), -- CONSTRAINT DeployedSHA_Length CHECK (length(DeployedSHA) = 40),
	DeployedBranch TEXT,
	RollbackSHA    CHAR(40), -- CONSTRAINT RollbackSHA_Length CHECK (length(RollbackSHA) = 40),
	RollbackBranch TEXT,
	PRIMARY KEY (Environment, Owner, Repository)
);

-- +goose StatementBegin
CREATE FUNCTION save_environment() RETURNS trigger AS
$$
	BEGIN
		IF new.State = 'success' THEN
  		-- Upsert environment, update rollback sha/branch.
			LOOP
		  	-- first try to update the key
			  UPDATE environments SET RollbackBranch = DeployedBranch, RollbackSHA = DeployedSHA WHERE Environment = new.Environment AND Owner = new.Owner AND Repository = new.Repository;
				IF found THEN
					EXIT;
				END IF;
				-- not there, so try to insert the key
				-- if someone else inserts the same key concurrently,
				-- we could get a unique-key failure
				BEGIN
					INSERT INTO environments(Environment, Owner, Repository, DeployedSHA, DeployedBranch, RollbackSHA, RollbackBranch) VALUES (new.Environment, new.Owner, new.Repository, new.SHA, new.Branch, new.SHA, new.Branch);
					EXIT;
				EXCEPTION WHEN unique_violation THEN
					-- Do nothing, and loop to try the UPDATE again.
				END;
			END LOOP;
		ELSIF new.State = 'failure' THEN
			-- Upsert environment, rollback deployed sha/branch. If inserting, null sha/branch
			LOOP
				-- first try to update the key
				UPDATE environments SET DeployedBranch = RollbackBranch, DeployedSHA = RollbackSHA WHERE Environment = new.Environment AND Owner = new.Owner AND Repository = new.Repository;
				IF found THEN
					EXIT;
				END IF;
				-- not there, so try to insert the key
				-- if someone else inserts the same key concurrently,
				-- we could get a unique-key failure
				BEGIN
					INSERT INTO environments(Environment, Owner, Repository, DeployedSHA, DeployedBranch, RollbackSHA, RollbackBranch) VALUES (new.Environment, new.Owner, new.Repository, NULL, NULL, NULL, NULL);
					EXIT;
				EXCEPTION WHEN unique_violation THEN
					-- Do nothing, and loop to try the UPDATE again.
				END;
			END LOOP;
		-- set deployed sha when deployment is created
    ELSIF new.State = 'unknown' THEN
			-- Upsert environment, set deployed sha/branch. If inserting, null rollback sha/branch
			LOOP
				-- first try to update the key
				UPDATE environments SET DeployedSHA = new.SHA, DeployedBranch = new.Branch WHERE Environment = new.Environment AND Owner = new.Owner AND Repository = new.Repository;
				IF found THEN
					EXIT;
				END IF;
				-- not there, so try to insert the key
				-- if someone else inserts the same key concurrently,
				-- we could get a unique-key failure
				BEGIN
					INSERT INTO environments(Environment, Owner, Repository, DeployedSHA, DeployedBranch, RollbackSHA, RollbackBranch) VALUES (new.Environment, new.Owner, new.Repository, new.SHA, new.Branch, NULL, NULL);
					EXIT;
				EXCEPTION WHEN unique_violation THEN
					-- Do nothing, and loop to try the UPDATE again.
				END;
			END LOOP;
		END IF;
		RETURN NULL;
	END;
$$ LANGUAGE plpgsql;
-- +goose StatementEnd

CREATE TRIGGER deploy_environment_insert
  AFTER INSERT ON deployments
	FOR EACH ROW
	EXECUTE PROCEDURE save_environment();

CREATE TRIGGER deploy_environment_update
  AFTER UPDATE ON deployments
	FOR EACH ROW
	-- don't update environment if state didn't change
	WHEN (old.State <> new.State)
	EXECUTE PROCEDURE save_environment();

-- +goose Down
-- SQL section 'Down' is executed when this migration is rolled back

DROP FUNCTION save_environment() CASCADE;
DROP TABLE environments;
