package webhook

import (
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/info"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"sync"
	"time"
)

var (
	dfinfo = info.NewInfo("deployfactory")
)

// DeployFactory interface for mocking
type DeployFactory interface {
	GetDeployer(string) deployment.Deployer
	ReturnDeployer(string)
	GetNumberOfActiveDeployers() int
	Stats() interface{}
}

type DeployFactoryConfig struct {
	// MaxConcurrentDeployers is maximum number of concurrent deploy jobs.
	MaxConcurrentDeployers int
	// ConsiderStallSecs is a number of seconds that's considered stalled
	// if deploy runs longer than this seconds.
	ConsiderStallSecs int
}

type DeployFactoryImpl struct {
	// Here we don't use the Deployer interface because we need to see the
	// internal variables for status checking
	deployers map[string]deployment.Deployer
	max       int
	stallsec  int
	mutex     sync.Mutex

	// Reference pointer to WC container
	wc *WebhookConfig
}

func NewDeployFactory(config *DeployFactoryConfig, wc *WebhookConfig) (DeployFactory, error) {
	// Create a DeployFactory for the return
	d := &DeployFactoryImpl{
		deployers: make(map[string]deployment.Deployer),
		max:       config.MaxConcurrentDeployers,
		stallsec:  config.ConsiderStallSecs,
		wc:        wc,
	}

	return d, nil
}

func (d *DeployFactoryImpl) GetDeployer(targetId string) deployment.Deployer {
	d.mutex.Lock()
	defer d.mutex.Unlock()

	log.Debug("requesting deployer ", targetId)
	if d.deployers[targetId] != nil {
		// do stall check
		jd := d.deployers[targetId]
		if jd.GetStartTime().IsZero() || int(time.Since(jd.GetStartTime()).Seconds()) < d.stallsec {
			log.Debugln("deployer target is running", targetId)
			dfinfo.IncreaseCounter("TotalTargetIsRunning")
			return nil
		}
		log.Printf("target %v is running longer than %d - allow next deploy", targetId, d.stallsec)
		dfinfo.IncreaseCounter("TotalStalledDeployers")
	}

	if len(d.deployers) >= d.max {
		log.Println("deployer queue is full:", len(d.deployers))
		dfinfo.IncreaseCounter("QueueFull")
		return nil
	}

	deployer := deployment.NewJenkinsDeployer(
		d.wc.GithubClient,
		d.wc.JenkinsClient,
		d.wc.StatsdClient,
		d.wc.ConsulPrefix,
	)
	d.deployers[targetId] = deployer

	// Update peak concurrent deployers
	num := int64(d.GetNumberOfActiveDeployers())
	if num > dfinfo.GetCounter("PeakConcurrentDeployers") {
		dfinfo.SetCounter("PeakConcurrentDeployers", num)
	}
	dfinfo.IncreaseCounter("TotalDeployersLaunched")

	return deployer
}

func (d *DeployFactoryImpl) ReturnDeployer(targetId string) {
	d.mutex.Lock()
	defer d.mutex.Unlock()
	delete(d.deployers, targetId)
	dfinfo.IncreaseCounter("TotalDeployersReturned")
}

func (d *DeployFactoryImpl) GetNumberOfActiveDeployers() int {
	return len(d.deployers)
}

func (d *DeployFactoryImpl) Stats() interface{} {
	type DeployStatus struct {
		Target    string
		State     string
		StartTime string
		Runtime   int
	}

	d.mutex.Lock()
	defer d.mutex.Unlock()
	s := &struct {
		Running int
		Jobs    []*DeployStatus
	}{
		Running: d.GetNumberOfActiveDeployers(),
		Jobs:    make([]*DeployStatus, 0, d.max),
	}

	for targetId, deployer := range d.deployers {
		s.Jobs = append(s.Jobs, &DeployStatus{
			Target:    targetId,
			State:     deployer.GetCurrentState(),
			StartTime: deployer.GetStartTime().String(),
			Runtime:   int(time.Since(deployer.GetStartTime()).Seconds()),
		})
	}

	return s
}

func GetDeployTargetId(event *deployment.Event) string {
	return fmt.Sprintf("%v/%v", *event.Repository.FullName, *event.Deployment.Environment)
}
