package webhook

import (
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/info"
	"encoding/json"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"io/ioutil"
	"net/http"
	"strings"
	"time"
)

var (
	qrinfo = info.NewInfo("queuerunner")
	dtinfo = info.NewInfo("deploytarget")
)

// queueDeployment is expected to be called within an http request handler. After preparing deployer goroutines,
// the event is queued and the http handler can respond
func (wc *WebhookConfig) queueDeployment(request *http.Request) error {
	jsonEvent, err := ioutil.ReadAll(request.Body)
	if err != nil {
		return fmt.Errorf("error queueing deployment: %v", err)
	}

	var event deployment.Event
	if err = json.Unmarshal(jsonEvent, &event); err != nil {
		return fmt.Errorf("error queueing deployment: %v", err)
	}

	// Add this deploy job to the queue and we're done.
	log.Debug("Enqueue a message")
	if err := wc.deployQueue.Push(string(jsonEvent)); err != nil {
		return err
	}
	return nil
}

func (wc *WebhookConfig) queueRunner() {
	for {
		// Pop an event
		data, err := wc.deployQueue.Pop(queueLongPullTimeSec)
		if err != nil {
			qrinfo.IncreaseCounter("Unreachable")
			log.Println("Queue is unreachable. Will try again:", err)
			time.Sleep(queueUnreachableRetryDelay)
			continue
		}
		if data == nil {
			continue
		}

		// Parse the event and targetId
		var event deployment.Event
		if err := json.Unmarshal([]byte(*data), &event); err != nil {
			qrinfo.IncreaseCounter("ParsingError")
			log.Println("Unable to parse event message:", err)
			continue
		}
		targetId := GetDeployTargetId(&event)

		// Check if there's any ongoing deploy job.
		// This logic is not very much necessary since actualy deploy will perform atomic check using
		// db transaction but we try to catch most of cases here.
		var deployer deployment.Deployer = nil
		isRunningTarget := isTargetRunningByOthers(targetId)
		if !isRunningTarget {
			// We only check local running jobs with memory, not from db to avoid stalled information
			// blocks new deploy. If there's any stalled work in any chance, it'll be taken care in the
			// process of deploy when it tries to create deploy status.
			deployer = wc.deployFactory.GetDeployer(targetId)
		}
		if isRunningTarget || deployer == nil {
			qrinfo.IncreaseCounter("Rescheduling")
			log.Printf("Rescheduling %v in %d sec. RunByOthers:%v", targetId, deployConflictRetryDelaySec, isRunningTarget)
			wc.deployQueue.PushWithDelay(deployConflictRetryDelaySec, *data)
			deployment.AppendLogGithubForEvent("Waiting on another ongoing deploy job to finish.", &event)
			continue
		}

		// Run the deploy job in background.
		go func(deployer deployment.Deployer, targetId string, event deployment.Event, data string) {
			log.Printf("run deployer. GitID=%v, TargetID=%v", *event.Deployment.ID, targetId)
			qrinfo.IncreaseCounter("RunDeployer")
			dtinfo.IncreaseCounter(targetId)
			dtinfo.SetProperty(targetId+"-LastStarted", time.Now().String())
			dtinfo.SetProperty(targetId+"-LastFinished", "in-progress")

			err := deployer.Deploy(&event)
			if err != nil {
				log.Printf("deployer error. GitID=%v, TargetID=%v - %v", *event.Deployment.ID, targetId, err)
				// This should be rare but could happen in a slim chance.
				if strings.Contains(err.Error(), "duplicate") {
					log.Printf("Runtime duplicatation found - Rescheduling %v in %d sec.", targetId, deployConflictRetryDelaySec)
					wc.deployQueue.PushWithDelay(deployConflictRetryDelaySec, data)
					deployment.AppendLogGithubForEvent("Waiting on another ongoing deploy job to finish.", &event)
					dtinfo.IncreaseCounter(targetId + "-DuplicateErrorCount")
				}
				dtinfo.IncreaseCounter(targetId + "-ErrorCount")
				dtinfo.SetProperty(targetId+"-LastErrorDetail", err.Error())
				dtinfo.SetProperty(targetId+"-LastErrorTime", time.Now().String())
			}

			dtinfo.SetProperty(targetId+"-LastFinished", time.Now().String())
			runsec := int64(time.Since(deployer.GetStartTime()).Seconds())
			dtinfo.SetCounter(targetId+"-LastRuntime", runsec)
			if runsec > dtinfo.GetCounter(targetId+"-LongestRuntime") {
				dtinfo.SetCounter(targetId+"-LongestRuntime", runsec)
			}

			wc.deployFactory.ReturnDeployer(targetId)
			log.Printf("end deployer. GitID=%v, TargetID=%v", *event.Deployment.ID, targetId)
		}(deployer, targetId, event, *data)
	}
}

func (wc *WebhookConfig) restartPendingJobs() {
	pjInfo := info.NewInfo("pendingjobs")
	pjInfo.SetProperty("StartedAt", time.Now().String())

	log.Debug("Restoring pending jobs")

	db := deployment.GetDB()
	if db == nil {
		// This can't happen but just to be safer.
		log.Fatal("Failed to get db object. Abort deploy resume process.")
	}
	queues, err := db.ListMyPendingJobs()
	if err != nil {
		log.Fatalf("Failed to list pending jobs - %v", err)
	}

	for _, queue := range queues {
		if *queue.Target == "" {
			continue
		}
		log.Printf("Restoring pending job. TargetID:%v, State:%v", *queue.Target, *queue.State)

		d := wc.deployFactory.GetDeployer(*queue.Target)
		if d != nil {
			go func(target string, dc string) {
				pjInfo.IncreaseCounter("Resume")
				if err := d.Resume(dc); err != nil {
					pjInfo.IncreaseCounter("ResumeError")
					log.Errorf("Error resuming %s - %v - %v", *queue.Target, err, dc)
				}
				wc.deployFactory.ReturnDeployer(target)
			}(*queue.Target, *queue.Deployer)
		} else {
			// This shouldn't be happening since pending job has priority.
			pjInfo.IncreaseCounter("ErrorOnGettingDeployer")
			log.Printf("Failed to get deployer. TargetID:%v - %v", *queue.Target, *queue.Deployer)

		}

		if err := db.DeleteMyDeploymentStatus(*queue.Target); err != nil {
			pjInfo.IncreaseCounter("ErrorOnDeletingStatus")
			log.Errorf("Failed to delete deploy status for %v - %v", *queue.Target, err)
		}
	}
	pjInfo.SetProperty("DoneAt", time.Now().String())
	log.Debug("Done restoring pending jobs")
}

func isTargetRunningByOthers(target string) bool {
	if db := deployment.GetDB(); db != nil {
		isRunningTarget, _ := db.IsTargetRunningByOthers(target)
		return isRunningTarget
	}
	//log.Warnf("DB hasn't setup, is this a local run?")
	return false
}
