package webhook

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"math"
	"net/http"
	"net/url"
	"path"
	"strconv"
	"strings"
	"text/template"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/config"
	"code.justin.tv/dta/skadi/pkg/deployment"
	"code.justin.tv/dta/skadi/pkg/email"
	"code.justin.tv/dta/skadi/pkg/subscriptions"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	"github.com/nlopes/slack"
)

// DeploymentStatusEvent contains information passed from the github webhook after a deployment is updated
type DeploymentStatusEvent struct {
	Deployment       *github.Deployment       `json:"deployment"`
	DeploymentStatus *github.DeploymentStatus `json:"deployment_status"`
	Repository       *github.Repository       `json:"repository"`
	Organization     *github.Organization     `json:"organization"`
	Sender           *github.User             `json:"sender"`
}

type deploymentStatus struct {
	DeployID              string
	DeployLinkURL         *url.URL
	DeploymentCreator     string
	DeploymentDescription string
	DeploymentEnvironment string
	DeploymentRef         string
	DeploymentStatus      string
	DeploymentSHA         string
	RepositoryOwner       string
	RepositoryName        string
	PayloadCodeReviewURL  string
	PayloadPreviousSHA    string
	PayloadSeverity       string
}

func readDeploymentStatusEvent(event *DeploymentStatusEvent) (*deploymentStatus, error) {
	status := &deploymentStatus{}

	deploy, err := deployment.FindDeployment(*event.Deployment.ID)
	if err != nil {
		return nil, err
	}
	status.DeployID = strconv.FormatInt(*deploy.ID, 10)

	if event.Deployment == nil {
		return nil, errors.New("deployment is nil")
	}
	if event.Deployment.Creator == nil {
		return nil, errors.New("deployment creator is nil")
	}
	if event.Deployment.Creator.Login == nil {
		return nil, errors.New("deployment creator login is nil")
	}
	status.DeploymentCreator = *event.Deployment.Creator.Login

	if deploy.Description != nil {
		status.DeploymentDescription = *deploy.Description
	} else if event.Deployment.Description != nil {
		// Unless we re-define FindDeployment() differently, runtime shouldn't
		// reach here, but this code stands here as a backup with no harm.
		status.DeploymentDescription = *event.Deployment.Description
		if len(status.DeploymentDescription) >= (math.MaxUint8 - 1) {
			status.DeploymentDescription += "... (truncated)"
		}
	} else {
		return nil, errors.New("deployment description is nil")
	}
	if event.Deployment.Environment == nil {
		return nil, errors.New("deployment environment is nil")
	}
	status.DeploymentEnvironment = *event.Deployment.Environment

	if event.Deployment.ID == nil {
		return nil, errors.New("deployment id is nil")
	}
	if event.Deployment.Ref == nil {
		return nil, errors.New("deployment ref is nil")
	}
	status.DeploymentRef = *event.Deployment.Ref
	if event.Deployment.SHA == nil {
		return nil, errors.New("deployment sha is nil")
	}
	status.DeploymentSHA = *event.Deployment.SHA

	if event.DeploymentStatus == nil {
		return nil, errors.New("deployment status is nil")
	}
	if event.DeploymentStatus.State == nil {
		return nil, errors.New("deployment status state is nil")
	}
	status.DeploymentStatus = *event.DeploymentStatus.State

	if event.Repository == nil {
		return nil, errors.New("repository is nil")
	}
	status.RepositoryName = *event.Repository.Name

	if event.Repository.Owner == nil {
		return nil, errors.New("repository owner is nil")

	}
	if event.Repository.Owner.Login == nil {
		return nil, errors.New("repository owner login is nil")
	}
	status.RepositoryOwner = *event.Repository.Owner.Login

	payload, err := deployment.ParsePayload(event.Deployment.Payload)
	if err != nil {
		return nil, err
	}
	status.PayloadCodeReviewURL = payload.CodeReviewURL
	status.PayloadPreviousSHA = payload.PreviousSHA
	status.PayloadSeverity = payload.Severity

	deployLinkURL := BaseURL
	deployLinkURL.Fragment = path.Join(status.RepositoryOwner, status.RepositoryName, "deploys", status.DeployID)
	status.DeployLinkURL = deployLinkURL
	return status, nil
}

func (status *deploymentStatus) Subject() string {
	return fmt.Sprintf("[deployment] - %v/%v: %v", status.RepositoryOwner, status.RepositoryName, strings.SplitN(status.DeploymentDescription, "\n", 2)[0])
}

type deploymentChanges struct {
	compareURL *url.URL
	commits    []github.RepositoryCommit
	err        error
}

func (changes *deploymentChanges) String() string {
	buffer := new(bytes.Buffer)
	if changes.err != nil {
		buffer.WriteString(changes.err.Error() + "\n")
	}
	if changes.compareURL != nil {
		buffer.WriteString(changes.compareURL.String() + "\n")
	}
	for _, commit := range changes.commits {
		if commit.Commit != nil && commit.Commit.Message != nil {
			// write only first line of commit messages to template
			buffer.WriteString("- " + extractTitle(*commit.Commit.Message, 76) + "\n")
		}
	}
	return buffer.String()
}

func extractTitle(message string, limit int) string {
	s := strings.Split(message, "\n")[0]
	if limit > 0 && len(s) > limit {
		s = s[0:limit] + "..."
	}
	return s
}

const (
	deploymentEmail = `
{{.DeploymentCreator}} deployed {{.RepositoryOwner}}/{{.RepositoryName}} to {{.DeploymentEnvironment}}

Risk Assessment: {{.PayloadSeverity}}
Status: {{.DeploymentStatus}}
{{with .PayloadCodeReviewURL}}
Code Review: {{.}}{{end}}

Changelog:
{{.Changes}}
{{with .DeployLinkURL}}Deploy Link: {{.}}{{end}}
Commit: https://git.xarth.tv/{{.RepositoryOwner}}/{{.RepositoryName}}/commit/{{.DeploymentRef}}
`
)

var (
	deploymentEmailTemplate *template.Template
)

func init() {
	deploymentEmailTemplate = template.Must(template.New("deployment").Parse(deploymentEmail))
}

func (wc *WebhookConfig) handleDeploymentStatus(request *http.Request) error {
	event := &DeploymentStatusEvent{}
	err := json.NewDecoder(request.Body).Decode(event)
	if err != nil {
		return err
	}

	status, err := readDeploymentStatusEvent(event)
	if err != nil {
		return err
	}

	log.Debug("handleDeploymentStatus(): event:", event)

	chatConfig, err := config.LoadChatConfig(wc.GithubClient, wc.ConsulClient, status.RepositoryOwner, status.RepositoryName, status.DeploymentSHA)
	if err != nil || chatConfig == nil {
		return fmt.Errorf("error loading chat config: %v", err)
	}
	if shouldBroadcastDeploymentStatus(event, chatConfig.Deploy) {
		err = wc.broadcastDeploymentStatus(status, chatConfig)
		if err != nil {
			return fmt.Errorf("error broadcasting deployment status: %v", err)
		}
	}

	environmentConfig, err := config.FindAppEnvironmentConfig(wc.GithubClient, wc.ConsulClient, status.RepositoryOwner, status.RepositoryName, status.DeploymentSHA, status.DeploymentEnvironment)
	if err != nil {
		return fmt.Errorf("error finding environment config: %v", err)
	}
	orgSubscriptions, err := subscriptions.GetSubscriptionsForEnv(&status.RepositoryOwner, &status.DeploymentEnvironment)
	if err != nil {
		return fmt.Errorf("error retrieving org subscriptions: %v", err)
	}
	if *orgSubscriptions != nil || environmentConfig.Subscriptions != nil || chatConfig.NotifyDeployUser == true {
		err = wc.notifyDeploymentStatus(status, environmentConfig, orgSubscriptions)
		if err != nil {
			return fmt.Errorf("error notifying deployment status: %v", err)
		}
	}

	return nil
}

func shouldBroadcastDeploymentStatus(event *DeploymentStatusEvent, config *api.ChatEnvironmentsConfig) bool {
	// specific env config
	c, ok := config.Environments[*event.Deployment.Environment]
	if ok {
		return shouldBroadcastEnvironment(event, c)
	}

	// wildcard branch config
	c, ok = config.Environments["*"]
	if ok {
		return shouldBroadcastEnvironment(event, c)
	}

	return false
}

func shouldBroadcastEnvironment(event *DeploymentStatusEvent, c api.ChatSectionConfig) bool {
	var b bool
	switch *event.DeploymentStatus.State {
	case "success":
		b = c.Success
	case "failure", "error":
		b = c.Failure
	case "pending":
		b = c.Pending
	}

	return b
}

func (wc *WebhookConfig) broadcastDeploymentStatus(status *deploymentStatus, chatConfig *api.ChatConfig) error {
	title := fmt.Sprintf("%v - %v", status.DeploymentStatus, status.Subject())
	var color string

	switch status.DeploymentStatus {
	case "pending":
		color = "warning"
	case "success":
		color = "good"
	case "error", "failure":
		color = "danger"
	}

	fields := []slack.AttachmentField{}
	if status.PayloadSeverity != "" {
		f := slack.AttachmentField{
			Title: "Risk Assessment",
			Value: status.PayloadSeverity,
			Short: false,
		}
		fields = append(fields, f)
	}
	text := fmt.Sprintf("%v deployed %v/%v@%v to %v", status.DeploymentCreator, status.RepositoryOwner, status.RepositoryName, status.DeploymentRef, status.DeploymentEnvironment)

	deployLinkURL := BaseURL
	deployLinkURL.Fragment = path.Join(status.RepositoryOwner, status.RepositoryName, "deploys", status.DeployID)

	slackMessageParams := slack.PostMessageParameters{
		Username: "skadi",
		IconURL:  "http://s.jtvnw.net/jtv_user_pictures/hosted_images/GlitchIcon_PurpleonWhite.png",
		Attachments: []slack.Attachment{
			slack.Attachment{
				Color: color,
				// AuthorName: *event.Deployment.Creator.Login,
				// AuthorIcon: *event.Deployment.Creator.AvatarURL,
				// AuthorLink: *event.Deployment.Creator.HTMLURL,
				Title:     title,
				TitleLink: status.DeployLinkURL.String(),
				Fields:    fields,
			},
		},
	}
	if chatConfig.Channel != nil {
		_, _, err := wc.SlackClient.PostMessage(*chatConfig.Channel, text, slackMessageParams)
		if err != nil {
			return err
		}
	}
	if chatConfig.NotifyDeployUser == true {
		log.Debug("notify_deploy_user is set")
		slackUserID, err := wc.getSlackUserID(status)
		if err != nil {
			// Just write a debug message if we can't send a slack notification for this user
			log.Warn(err)
		} else if slackUserID == "" {
			log.Warn("Unable to determine the Slack userid for deploy creator: ", status.DeploymentCreator)
		} else {
			_, _, err := wc.SlackClient.PostMessage(slackUserID, text, slackMessageParams)
			if err != nil {
				return err
			}
		}
	} else {
		log.Debug("notify_deploy_user not set")
	}

	return nil
}

func (wc *WebhookConfig) notifyDeploymentStatus(status *deploymentStatus, environmentConfig *api.AppEnvironmentConfig, orgSubscriptions *[]subscriptions.Subscription) error {
	//store all subscribers in a map, so we don't send to dupes
	var allSubscriptions = make(map[string]map[string]bool)
	var recipients []string

	for _, sub := range *orgSubscriptions {
		bools := make(map[string]bool)
		bools["success"] = sub.Success
		bools["pending"] = sub.Pending
		bools["failure"] = sub.Failure
		if _, ok := allSubscriptions[sub.Email]; ok {
			if sub.Environment == "" {
				continue
			}
		}
		allSubscriptions[sub.Email] = bools
	}

	//let the settings in deploy.json override org-wide settings
	for address, sub := range environmentConfig.Subscriptions {
		bools := make(map[string]bool)
		bools["success"] = sub.Success
		bools["pending"] = sub.Pending
		bools["failure"] = sub.Failure
		allSubscriptions[address] = bools
	}
	for address, subscriptions := range allSubscriptions {
		var b bool
		switch status.DeploymentStatus {
		case "success":
			b = subscriptions["success"]
		case "failure", "error":
			b = subscriptions["failure"]
		case "pending":
			b = subscriptions["pending"]
		default:
			continue
		}
		if b {
			recipients = append(recipients, address)
		}
	}
	if len(recipients) == 0 {
		return nil
	}

	changes, err := wc.getDeploymentChanges(status)
	if err != nil {
		return err
	}

	textBuf := new(bytes.Buffer)
	err = deploymentEmailTemplate.Execute(textBuf, &struct {
		*deploymentStatus
		Changes *deploymentChanges
	}{status, changes})
	if err != nil {
		return err
	}

	err = email.SendEmail(&email.Email{
		Recipients: recipients,
		Subject:    status.Subject(),
		Text:       textBuf.String(),
	})
	return err
}

// getDeploymentCreatorEmail looks up user information in github to create a
// From: header. Name will default to github login if it cannot be looked up,
// and email will default to mailDefaultAddress.
func (wc *WebhookConfig) getDeploymentCreatorEmail(status *deploymentStatus) string {
	user, _, _ := wc.GithubClient.Users.Get(context.TODO(), status.DeploymentCreator)

	var name, address string
	if user == nil || user.Name == nil || *user.Name == "" {
		name = status.DeploymentCreator
	} else {
		name = *user.Name
	}
	if user == nil || user.Email == nil || *user.Email == "" {
		address = mailDefaultAddress
	} else {
		address = *user.Email
	}

	return fmt.Sprintf("%s <%s>", name, address)
}

func (wc *WebhookConfig) getDeploymentChanges(status *deploymentStatus) (*deploymentChanges, error) {
	changes := &deploymentChanges{}
	if status.PayloadPreviousSHA == status.DeploymentSHA {
		changes.err = errors.New("SHA is identical to previous deployment")
	} else {
		compareCommits, response, err := wc.GithubClient.Repositories.CompareCommits(context.TODO(), status.RepositoryOwner, status.RepositoryName, status.PayloadPreviousSHA, status.DeploymentSHA)
		if err != nil {
			changes.err = err
		}
		compareURL := response.Request.URL.String()
		compareURL = strings.Replace(compareURL, "/api/v3/repos/", "/", -1)
		changes.compareURL, err = url.Parse(compareURL)
		if err != nil {
			return nil, err
		}
		changes.commits = compareCommits.Commits
	}
	return changes, nil
}

// Try to determine the slack user ID for the person triggering the deployment
// We try to match based on the following in order of preference
// Github email == Slack email
// Github user full name = Slack full name
//
func (wc *WebhookConfig) getSlackUserID(status *deploymentStatus) (string, error) {
	creatorUsername := status.DeploymentCreator
	creatorJTVEmail := status.DeploymentCreator + "@justin.tv" // Typically used in slack for email.  Maybe a standard? not sure.
	creatorTTVEmail := status.DeploymentCreator + "@twitch.tv"
	creatorNameEmail := wc.getDeploymentCreatorEmail(status)                              // This returns something like: John Alberts <johnalb@justin.tv>
	creatorName := strings.Split(creatorNameEmail, " <")[0]                               // Just the name
	creatorEmail := strings.Replace(strings.Split(creatorNameEmail, " <")[1], ">", "", 1) // Just the email

	// Check if this is a special user
	overrideID, err := getSlackOverideID(creatorUsername)
	if err != nil || overrideID != "" {
		// This is a special user, so just return
		return overrideID, err
	}

	// Get a list of users so we can search through them
	slackUsers, err := wc.SlackClient.GetUsers()
	if err != nil {
		return "", err
	}

	// TODO: search list of slackUsers for RealName = full name of deployment user
	for _, v := range slackUsers {
		// if the email address matches, we're done.
		if v.Profile.Email == creatorEmail {
			return v.ID, nil
		}
	}

	for _, v := range slackUsers {
		if v.Profile.Email == creatorJTVEmail || v.RealName == creatorName || v.Profile.Email == creatorTTVEmail {
			return v.ID, nil
		}
	}
	return "", errors.New("Unable to determine slack user id for deploymentCreator: " + creatorUsername)
}

// Checks if a special slackid exists for the user
// Returns the slackid if the user has an override, otherwise returns empty string
// Returns an error if the user is a special user that shouldn't receive slack messages.
func getSlackOverideID(userName string) (string, error) {
	var overrideUserList = map[string]string{
		"devtools":          "NOID",
		"mbollier":          "U03C7EY0Y",
		"hassaan-markhiani": "U03SWDZL7",
		"tiffany-huang":     "U03SWEBCK",
		"xangold":           "U03T6UVF9",
		"jos":               "U03T6MDNZ",
		"dlu":               "U03SWE1BP",
		"mixonic":           "U0MP9NBRD",
		"rwjblue":           "U0J6METJP",
		"achou":             "U0TFV834H",
		"seph":              "U0FLSJVBP",
		"bantic":            "U0NMGECQ6",
		"ben-swartz":        "U03SZ7P1T",
		"hank":              "U0W0K8TED",
		"Kai-Hayashi":       "U03SZ0HE3",
		"abrown":            "U1W7UTPL4",
		"jamesjia":          "U2K8CU1NY",
		"toph":              "U0AAURCP9",
		"tiffache":          "U1LMDAJPN",
	}
	overrideUser := overrideUserList[userName]
	if overrideUser == "NOID" {
		return "", errors.New("user is a special user")
	}
	return overrideUser, nil
}
