package webhook

import (
	"testing"

	"code.justin.tv/dta/skadi/api"
	"github.com/google/go-github/github"
)

func deployStatus(state, env string) *DeploymentStatusEvent {
	return &DeploymentStatusEvent{
		Deployment: &github.Deployment{
			Environment: &env,
		},
		DeploymentStatus: &github.DeploymentStatus{
			State: &state,
		},
	}
}

func deployConfig(env string, p, s, f bool) *api.ChatEnvironmentsConfig {
	return &api.ChatEnvironmentsConfig{
		Environments: map[string]api.ChatSectionConfig{
			env: {
				Success: s,
				Pending: p,
				Failure: f,
			},
		},
	}
}

func TestDeploymentSubject(t *testing.T) {
	owner := "owner"
	name := "name"
	description := "Testing description"
	status := &deploymentStatus{RepositoryOwner: owner, RepositoryName: name, DeploymentDescription: description}
	subject := status.Subject()
	expected := "[deployment] - owner/name: Testing description"
	if subject != expected {
		t.Errorf("&status{...}.Subject() = %s; want %s", subject, expected)
	}
}

func TestDeploymentSubjectNewlines(t *testing.T) {
	owner := "owner"
	name := "name"
	description := "Testing description\nwith newlines"
	status := &deploymentStatus{RepositoryOwner: owner, RepositoryName: name, DeploymentDescription: description}
	subject := status.Subject()
	expected := "[deployment] - owner/name: Testing description"
	if subject != expected {
		t.Errorf("&status{...}.Subject() = %s; want %s", subject, expected)
	}
}

func TestShouldBroadcastDeploymentStatus(t *testing.T) {
	for i, set := range []struct {
		event    *DeploymentStatusEvent
		config   *api.ChatEnvironmentsConfig
		expected bool
	}{
		{deployStatus("pending", "production"), &api.ChatEnvironmentsConfig{}, false},
		{deployStatus("success", "production"), deployConfig("production", false, true, false), true},
		{deployStatus("failure", "staging-2"), deployConfig("*", false, false, true), true},
	} {
		if shouldBroadcastDeploymentStatus(set.event, set.config) != set.expected {
			t.Fatalf("unexpected status broadcast. Set Index: %v", i)
		}
	}
}

func TestGetSlackOverideID(t *testing.T) {

	// test a special user
	devtoolsID, err := getSlackOverideID("devtools")
	if err == nil || devtoolsID != "" {
		t.Fatal("failed to find special 'devtools' slack user")
	}

	// test a user in the list
	// Note: this test user will need to change if rwjblue ever gets a new job. :)
	userID, err := getSlackOverideID("rwjblue")
	if err != nil || userID != "U0J6METJP" {
		t.Fatal("error retrieving slackID for user 'rwjblue'")
	}

	// test a user not in the list
	missingID, err := getSlackOverideID("lkjasdkfljafkj")
	if err != nil || missingID != "" {
		t.Fatal("invalid slackID returned for random user")
	}

}
