package webhook

import (
	"bytes"
	"code.justin.tv/dta/skadi/pkg/deployment"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"net/http"
	"sync"
	"testing"
	"time"
)

const (
	jobRuntime  time.Duration = (50 * time.Millisecond)
	waitTimeout float64       = 3.0
)

func init() {
	log.SetLevel(log.WarnLevel)
}

func TestQueueRunner_Basic(t *testing.T) {
	// Basic pickup test
	wc := newWebhookConfig(10)
	go wc.queueRunner()
	wc.queueDeployment(generateDeploymentRequest(0, "staging1"))
	monitorQueueRunner(t, wc, 1, waitTimeout)
}

func TestQueueRunner_ConcurrencyControl(t *testing.T) {
	// Test maximum concurrent deploy
	wc := newWebhookConfig(3)
	go wc.queueRunner()
	wc.queueDeployment(generateDeploymentRequest(0, "staging2"))
	wc.queueDeployment(generateDeploymentRequest(0, "staging3"))
	wc.queueDeployment(generateDeploymentRequest(0, "staging4"))
	wc.queueDeployment(generateDeploymentRequest(0, "staging5"))
	monitorQueueRunner(t, wc, 3, waitTimeout)
}

func TestQueueRunner_Serialization(t *testing.T) {
	// Test same target serialization
	wc := newWebhookConfig(10)
	go wc.queueRunner()
	wc.queueDeployment(generateDeploymentRequest(0, "staging1"))
	wc.queueDeployment(generateDeploymentRequest(0, "staging2"))
	wc.queueDeployment(generateDeploymentRequest(0, "staging1"))
	monitorQueueRunner(t, wc, 2, waitTimeout)
}

func newWebhookConfig(maxConcurency int) *WebhookConfig {
	wc := &WebhookConfig{
		deployQueue: &MockDeployQueue{},
	}
	wc.deployFactory = &MockDeployFactory{
		f: &DeployFactoryImpl{
			deployers: make(map[string]deployment.Deployer),
			max:       maxConcurency,
			stallsec:  1,
			wc:        wc,
		},
	}
	return wc
}

func monitorQueueRunner(t *testing.T, wc *WebhookConfig, maxConcurency int, maxRunTime float64) {
	peak := 0
	for start := time.Now(); ; {
		if time.Since(start).Seconds() > maxRunTime {
			t.Error("active deployers: expect 0 but", wc.deployFactory.GetNumberOfActiveDeployers())
			break
		}
		n := wc.deployFactory.GetNumberOfActiveDeployers()
		if n > peak {
			peak = n
		}
		if 0 == wc.deployQueue.Len() && 0 == n && peak > 0 {
			break
		}
	}
	if peak != maxConcurency {
		t.Error("max concurrency: expect", maxConcurency, "but", peak)
	}
}

//
// MockDeployFactory calls full logic of real DeployFactoryImpl but returns
// MockDeployer for the actual deploy worker.
//
type MockDeployFactory struct {
	f DeployFactory
}

func (d *MockDeployFactory) GetDeployer(targetId string) deployment.Deployer {
	if r := d.f.GetDeployer(targetId); r == nil {
		return nil
	}
	return &MockDeployer{}
}

func (d *MockDeployFactory) ReturnDeployer(targetId string) {
	d.f.ReturnDeployer(targetId)
}

func (d *MockDeployFactory) GetNumberOfActiveDeployers() int {
	return d.f.GetNumberOfActiveDeployers()
}

func (d *MockDeployFactory) Stats() interface{} {
	return nil
}

//
// MockDeployer mimics Jenkins deploy job
//
type MockDeployer struct {
	StartTime    time.Time
	CurrentState string
}

func (d *MockDeployer) Deploy(event *deployment.Event) error {
	d.StartTime = time.Now()
	d.CurrentState = "setup"
	time.Sleep(jobRuntime)
	d.CurrentState = "end"
	return nil
}

func (d *MockDeployer) Reset() {
	d.CurrentState = "setup"
}

func (d *MockDeployer) Resume(key string) error {
	return nil
}

func (d *MockDeployer) GetStartTime() time.Time {
	return d.StartTime
}

func (d *MockDeployer) GetCurrentState() string {
	return d.CurrentState
}

//
// MockDeployQueue mimics SQS queue behaviors
//
type MockDeployQueue struct {
	mutex sync.Mutex
	data  []string
}

func (q *MockDeployQueue) Len() int64 {
	q.mutex.Lock()
	defer q.mutex.Unlock()
	return int64(len(q.data))
}
func (q *MockDeployQueue) Push(message string) error {
	return q.PushWithDelay(0, message)
}

func (q *MockDeployQueue) PushWithDelay(delaySec int, message string) error {
	if delaySec > 0 {
		go func() {
			time.Sleep(1 * time.Millisecond)
			q.mutex.Lock()
			q.data = append(q.data, message)
			q.mutex.Unlock()
		}()
	} else {
		q.mutex.Lock()
		q.data = append(q.data, message)
		q.mutex.Unlock()
	}
	log.Debugf("PUSH: %v", message)

	return nil
}

func (q *MockDeployQueue) Pop(pullsec int) (*string, error) {
	q.mutex.Lock()
	defer q.mutex.Unlock()
	if len(q.data) == 0 {
		return nil, nil
	}
	value := q.data[0]
	q.data = q.data[1:]
	log.Debugf("POP: %v", value)
	return &value, nil
}

func (q *MockDeployQueue) Purge() error {
	q.mutex.Lock()
	defer q.mutex.Unlock()
	q.data = []string{}
	return nil
}

func generateEvent(id int, env string) string {
	return fmt.Sprintf(`
{
        "deployment": {
                "id": %v,
                "environment": "%v"
        },
        "repository": {
                "full_name": "dta/test"
        }
}
`, id, env)
}

func generateDeploymentRequest(id int, env string) *http.Request {
	json := generateEvent(id, env)
	req, _ := http.NewRequest("POST", "foo", bytes.NewBufferString(json))
	return req
}
