package webhook

import (
	"code.justin.tv/common/solidsqs"
	"code.justin.tv/dta/skadi/pkg/info"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"io"
	"os"
	"time"
)

const (
	queueLongPullTimeSec        = 5
	queueUnreachableRetryDelay  = (5 * time.Second)
	deployConflictRetryDelaySec = 60
)

var (
	dqinfo = info.NewInfo("deployqueue")
)

// Queue interface for mocking
type Queue interface {
	Len() int64
	Push(...string) (int, error)
	PushWithDelay(int, ...string) (int, error)
	Pop(int, int) ([]string, error)
	Purge() error
}

// DeployQueue interface for mocking
type DeployQueue interface {
	Push(string) error
	PushWithDelay(int, string) error
	Pop(int) (*string, error)
	Len() int64
}

type DeployQueueImpl struct {
	queue Queue
}

type DeployQueueConfig struct {
	// QueueURL is AWS SQS's url of the deploy queue.
	QueueURL string
	// QueueRegion is the AWS region code of the AWS SQS.
	QueueRegion string
	// EnableQueueLog enables queue activity log.
	EnableQueueLog bool
	// QueueLogPath is a file path for the queue activity log.
	QueueLogPath string
	// QueueRetryIntervalMin is minimum interval for queue retry in Millisecond
	QueueRetryIntervalMin time.Duration
	// QueueRetryIntervalMax is maximum interval for queue retry in Millisecond
	QueueRetryIntervalMax time.Duration
	// QueueRetryRuntime is maximum retry runtime in Millisecond.
	QueueRetryRuntime time.Duration
}

func NewDeployQueue(config *DeployQueueConfig) (DeployQueue, error) {
	// Enable queue access log if desired
	var out io.Writer
	if config.EnableQueueLog {
		if o, err := os.OpenFile(config.QueueLogPath, os.O_APPEND|os.O_CREATE|os.O_WRONLY, 0644); err != nil {
			log.Println("Could not open queue log at", config.QueueLogPath)
		} else {
			out = io.Writer(o)
		}
	}

	// Create a new SQS object
	queue, err := solidsqs.New(config.QueueURL,
		&aws.Config{Region: aws.String(config.QueueRegion)},
		&solidsqs.Config{
			RetryIntervalMin:   config.QueueRetryIntervalMin,
			RetryIntervalMax:   config.QueueRetryIntervalMax,
			RetryRuntime:       config.QueueRetryRuntime,
			AsynchronousDelete: false,
			DebugOut:           out,
		})
	if err != nil {
		return nil, err
	}

	// Test queue connection
	if queue.Len() < 0 {
		return nil, fmt.Errorf("Queue is unreachable. %s", config.QueueURL)
	}

	dqinfo.IncreaseCounter("New")
	return &DeployQueueImpl{
		queue: queue,
	}, nil
}

func (q *DeployQueueImpl) Push(data string) error {
	if num, err := q.queue.Push(data); err != nil || num != 1 {
		dqinfo.IncreaseCounter("PushError")
		return err
	}
	dqinfo.IncreaseCounter("Push")
	return nil
}

func (q *DeployQueueImpl) PushWithDelay(delaySec int, data string) error {
	if num, err := q.queue.PushWithDelay(delaySec, data); err != nil || num != 1 {
		dqinfo.IncreaseCounter("PushDelayError")
		return err
	}
	dqinfo.IncreaseCounter("PushDelay")
	return nil
}

func (q *DeployQueueImpl) Pop(pullSec int) (*string, error) {
	dqinfo.IncreaseCounter("PopAttempt")

	data, err := q.queue.Pop(1, pullSec)
	if err != nil {
		dqinfo.IncreaseCounter("PopError")
		return nil, err
	}
	if len(data) == 0 {
		return nil, nil
	}
	dqinfo.IncreaseCounter("Pop")
	return &data[0], nil
}

func (q *DeployQueueImpl) Len() int64 {
	dqinfo.IncreaseCounter("Len")
	return q.queue.Len()
}
