package webhook

import (
	"code.justin.tv/dta/skadi/pkg/info"
	"encoding/json"
	"fmt"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	"io/ioutil"
	"net/http"
	"path"
	"time"
)

const (
	jenkinsJobFile              = "jenkins.groovy"
	jenkinsJobName              = "job-dsl-seed"
	jenkinsPayloadSizeThreshold = (13 * 1024)
)

var (
	jjinfo = info.NewInfo("job-dsl-seed")
)

func (wc *WebhookConfig) handlePushStatus(req *http.Request) error {
	// TODO: After github lib update, re-write the decoding logic.
	// For debugging purpose.
	body, err := ioutil.ReadAll(req.Body)
	if err != nil {
		return fmt.Errorf("failed to read request body - %v", err)
	}

	var event github.WebHookPayload
	json.Unmarshal(body, &event)
	// Currently there's a known error happening due to mismatch of github lib version with running GHE
	// So we don't check error at decode time but check the structure members instead
	if event.Ref == nil {
		// This is the absolute decoding fail case, let's lot it here
		log.Printf("Unable to parse push message body - %v - %v", err, string(body))
		return fmt.Errorf("nil ref; can't proceed")
	}

	if event.Repo == nil || event.Repo.DefaultBranch == nil {
		// I don't know how it would be possible to hit this but just in case:
		return fmt.Errorf("can't find default branch")
	}

	// Only process look for jenkinsJobFile on the default branch:
	if path.Base(*event.Ref) != *event.Repo.DefaultBranch {
		return nil
	}

	// Do sanity check
	if _, err := json.Marshal(event); err != nil {
		return fmt.Errorf("Failed to marshal event body. - %v", err)
	}

	// Run jenkins job if qualified
	for _, commit := range event.Commits {
		if containsJenkinsJobFile(
			commit.Modified,
			commit.Removed,
			commit.Added,
		) {

			// Make a payload from the simplified event
			payload, err := json.Marshal(checkAndShrinkEvent(&event))
			if err != nil {
				return fmt.Errorf("Failed to marshal simplified event body. - %v", err)
			}

			jjinfo.IncreaseCounter("Call")
			jjinfo.SetProperty("LastCallAt", time.Now().String())

			// It doesn't matter if we succeed or fail we are done here.
			// Just return the error.
			params := make(map[string]string)
			params["payload"] = string(payload)
			_, err = wc.JenkinsClient.BuildWithParameters(jenkinsJobName, params)
			if err != nil {
				jjinfo.IncreaseCounter("Error")
				jjinfo.SetProperty("LastError", err.Error())
				jjinfo.SetProperty("LastErrorAt", time.Now().String())
				return fmt.Errorf("Failed to trigger jenkins job - %v", err)
			}
			break
		}
	}

	return nil
}

// containsJenkinsJobFile is a small helper function that returns true if the
// list contains the jenkinsJobFile.
func containsJenkinsJobFile(fileSets ...[]string) bool {
	for _, files := range fileSets {
		for _, file := range files {
			if path.Base(file) == jenkinsJobFile {
				return true
			}
		}
	}

	return false
}

// DTA-1886, DTA-2256 limit size of payload sent to jenkins if the size if too large
func checkAndShrinkEvent(event *github.WebHookPayload) *github.WebHookPayload {
	// Check the size of payload
	if payload, err := json.Marshal(event); err != nil || len(payload) < jenkinsPayloadSizeThreshold {
		return event
	}

	// Shrink
	event.Commits = []github.WebHookCommit{
		github.WebHookCommit{
			Added: []string{jenkinsJobFile},
		},
	}
	event.HeadCommit = nil

	return event
}
