package webhook

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/dta/skadi/api"
	"code.justin.tv/dta/skadi/pkg/config"
	log "github.com/Sirupsen/logrus"
	"github.com/google/go-github/github"
	"github.com/nlopes/slack"
)

// CommitStatusEvent contains information passed from the github webhook after a jenkins build marks a commit good or bad or other kinds of events like coverage
type CommitStatusEvent github.StatusEvent

func (event *CommitStatusEvent) isCoverageEvent() bool {
	return event.Sender != nil && event.Sender.Login != nil && *event.Sender.Login == "codecov"
}

func (event *CommitStatusEvent) isBuildEvent() bool {
	return !event.isCoverageEvent()
}

func (event *CommitStatusEvent) BranchNames() []string {
	branchNames := []string{}
	for _, branch := range event.Branches {
		if branch.Name != nil {
			branchNames = append(branchNames, *branch.Name)
		}
	}

	return branchNames
}

func (wc *WebhookConfig) handleCommitStatus(request *http.Request) error {
	var event CommitStatusEvent
	err := json.NewDecoder(request.Body).Decode(&event)
	if err != nil {
		return err
	}

	err = wc.broadcastStatus(&event)
	if err != nil {
		return fmt.Errorf("error broadcasting commit status for %v: %v", *event.Name, err)
	}

	if event.isBuildEvent() {
		err = wc.reportBuildLag(&event)
		if err != nil {
			log.Printf("error reporting build lag %v", err)
		}
	}
	return nil
}

func shouldBroadcastCommitStatus(event *CommitStatusEvent, config *api.ChatBranchesConfig) bool {
	// specific branch config
	branches := event.BranchNames()
	for _, branch := range branches {
		c, ok := config.Branches[branch]
		if !ok {
			continue
		}
		return shouldBroadcastBranch(event, c)
	}

	// wildcard branch config
	c, ok := config.Branches["*"]
	if ok {
		return shouldBroadcastBranch(event, c)
	}

	return false
}

func shouldBroadcastBranch(event *CommitStatusEvent, c api.ChatSectionConfig) bool {
	var b bool
	switch *event.State {
	case "success":
		b = c.Success
	case "failure", "error":
		b = c.Failure
	case "pending":
		b = c.Pending
	}

	return b
}

// There are three characters you must convert to HTML entities and only three: &, <, and >.
//
// Slack will take care of the rest.
//
// * Replace the ampersand, &, with &amp;
// * Replace the less-than sign, < with &lt;
// * Replace the greater-than sign, > with &gt;
//
// https://api.slack.com/docs/message-formatting#how_to_escape_characters
var slackEscaper = strings.NewReplacer(
	`&`, "&amp;",
	`<`, "&lt;",
	`>`, "&gt;",
)

func formatCommitMessage(commitURL, commit string) string {
	msg := slackEscaper.Replace(commit)
	lines := strings.Split(msg, "\n")
	text := fmt.Sprintf("<%v|%v>", commitURL, lines[0])
	if len(lines) > 1 {
		text += "\n" + strings.Join(lines[1:], "\n")
	}

	return text
}

func (wc *WebhookConfig) broadcastStatus(event *CommitStatusEvent) error {
	chatConfig, err := config.LoadChatConfig(wc.GithubClient, wc.ConsulClient, *event.Repo.Owner.Login, *event.Repo.Name, *event.SHA)
	if err != nil || chatConfig == nil {
		return fmt.Errorf("error loading chat config: %v", err)
	}
	if chatConfig.Channel == nil {
		return nil
	}
	sp := NewSlackPayload(event, chatConfig)
	if !sp.shouldBroadcast() {
		return nil
	}
	_, _, err = wc.SlackClient.PostMessage(*chatConfig.Channel, "", sp.Message())
	return err
}

// reportBuildLag measures the time between when the commit ocurred and when the build succeeded.
// aims to bring to light commit-to-deployable time which is larger than build time
func (wc *WebhookConfig) reportBuildLag(event *CommitStatusEvent) error {
	// Possible event states from github are: pending, success, failure and error
	// We don't care about metrics for the pending status.
	if *event.State == "pending" {
		return nil
	}

	now := time.Now()
	commitTime := *event.Commit.Commit.Committer.Date

	bucket := fmt.Sprintf(
		"build-lag.%s.%s",
		strings.Replace(*event.Name, "/", ".", 1), *event.State)
	wc.StatsdClient.Incr(bucket)
	wc.StatsdClient.Timing(bucket, now.Sub(commitTime))

	return nil
}

func NewSlackPayload(event *CommitStatusEvent, chatConfig *api.ChatConfig) SlackPayload {
	return SlackPayload{
		event:      event,
		chatConfig: chatConfig,
	}
}

type SlackPayload struct {
	event      *CommitStatusEvent
	chatConfig *api.ChatConfig
}

func (sp SlackPayload) shouldBroadcast() bool {
	if sp.event.isCoverageEvent() && !shouldBroadcastCommitStatus(sp.event, sp.chatConfig.Coverage) {
		return false
	}
	if sp.event.isBuildEvent() && !shouldBroadcastCommitStatus(sp.event, sp.chatConfig.Build) {
		return false
	}
	return true

}

func (sp SlackPayload) color() string {
	if sp.event.State != nil {
		switch *sp.event.State {
		case "success":
			return "good"
		case "failure", "error":
			return "danger"
		case "pending":
			return "warning"
		}
	}
	return "#dcdcdc"
}

func (sp SlackPayload) branches() string {
	return strings.Join(sp.event.BranchNames(), ",")
}

func (sp SlackPayload) title() string {
	var title string
	if sp.event.isCoverageEvent() {
		title = "Coverage"
	} else {
		title = "Build"
	}
	if sp.event.State != nil {
		switch *sp.event.State {
		case "success":
			title += " Success"
		case "failure", "error":
			title += " Failure"
		case "pending":
			title += " Starting..."
		}
	}

	// print the coverage reason for the event
	// example: "58.93% (target 25%)"
	if sp.event.isCoverageEvent() && sp.event.Description != nil {
		title += " - " + *sp.event.Description
	}
	return title
}

func (sp SlackPayload) titleLink() string {
	if sp.event.TargetURL != nil {
		return *sp.event.TargetURL
	}
	return "unknown"
}

func (sp SlackPayload) commitURL() string {
	if sp.event.Commit != nil && sp.event.Commit.HTMLURL != nil {
		return *sp.event.Commit.HTMLURL
	}
	if sp.event.Repo != nil && sp.event.Repo.HTMLURL != nil && sp.event.SHA != nil {
		return fmt.Sprintf("%v/commit/%v", *sp.event.Repo.HTMLURL, *sp.event.SHA)
	}
	return ""
}

func (sp SlackPayload) commitMessage() string {
	return *sp.event.Commit.Commit.Message
}

func (sp SlackPayload) text() string {
	return formatCommitMessage(sp.commitURL(), sp.commitMessage())
}

func (sp SlackPayload) repository() string {
	if sp.event.Repo != nil && sp.event.Repo.FullName != nil {
		return *sp.event.Repo.FullName
	}
	if sp.event.Name != nil {
		return *sp.event.Name
	}
	return ""
}

func (sp SlackPayload) author() *github.User {
	if sp.event.Commit != nil {
		if sp.event.Commit.Committer != nil {
			return sp.event.Commit.Committer
		}
		if sp.event.Commit.Author != nil {
			return sp.event.Commit.Author
		}
	}
	return nil
}

func (sp SlackPayload) authorName() string {
	author := sp.author()
	if author != nil && author.Login != nil {
		return *author.Login
	}
	return "unknown"
}

func (sp SlackPayload) authorIcon() string {
	author := sp.author()
	if author != nil && author.AvatarURL != nil {
		return *author.AvatarURL
	}
	return ""
}

func (sp SlackPayload) authorLink() string {
	author := sp.author()
	if author != nil && author.HTMLURL != nil {
		return *author.HTMLURL
	}
	return ""
}

func (sp SlackPayload) Message() slack.PostMessageParameters {
	return slack.PostMessageParameters{
		Username: "skadi",
		IconURL:  "http://s.jtvnw.net/jtv_user_pictures/hosted_images/GlitchIcon_PurpleonWhite.png",
		Attachments: []slack.Attachment{
			slack.Attachment{
				Title:      sp.title(),
				TitleLink:  sp.titleLink(),
				Color:      sp.color(),
				AuthorName: sp.authorName(),
				AuthorIcon: sp.authorIcon(),
				AuthorLink: sp.authorLink(),
				Text:       sp.text(),
				Fields: []slack.AttachmentField{
					{
						Title: "Repository",
						Value: sp.repository(),
						Short: true,
					},
					{
						Title: "Branches",
						Value: sp.branches(),
						Short: true,
					},
				},
			},
		},
	}
}
